-----------------------------------------------------------------------------------------------
-- Client Lua Script for Jabbithole
-- Copyright (c) NCsoft. All rights reserved
-----------------------------------------------------------------------------------------------
 
require "Window"
require "GameLib"
require "Apollo"
require "Spell"
require "Unit"
require "PublicEventsLib"
require "ChallengesLib"
require "PlayerPathLib"
require "PathMission"

-----------------------------------------------------------------------------------------------
-- Jabbithole Module Definition
-----------------------------------------------------------------------------------------------
local Jabbithole = {} 
local JSON = OBJDEF:new()
local VERSION = 1000001003
 
-----------------------------------------------------------------------------------------------
-- Constants
-----------------------------------------------------------------------------------------------
 
local droppedItems = {}
local unitCache = {}
local justLootedCache = {}
local inventoryCache = {}
local tVitals = Unit.GetVitalTable()
local creatureLocationCached = {}
local containerCache = {}
local reverseContainerCache = {}
local teachesCache = {}
local reverseTeachesCache = {}
local dyeCache = {}
local reverseDyeCache = {}
local salvageCache = {}
local reverseSalvageCache = {}
local lastDatacube = nil
local bVeteranMode = false
local deprecates = 7
local lastTradeskillSaveTime = 0
local lastUploadWarning = 0
local rootProcessed = {}

local KEY_ITEMS="items"
local KEY_ITEM_PROF_REQ="itempr"
local KEY_ITEM_SLOT="itemsl"
local KEY_ITEM_TYPE="itemtp"
local KEY_ITEM_FAMILY="itemf"
local KEY_ITEM_CATEGORY="itemcat"
local KEY_QUEST_CATEGORY="questc"
local KEY_QUEST_EPISODE="queste"
local KEY_QUESTS="quest"
local KEY_ZONES="zone"
local KEY_CREATURES="npc"
local KEY_TITLES="titles2"
local KEY_SPELLS="spells"
local KEY_CREATURE_SPELL="npcspell"
local KEY_VENDOR_ITEM="vnditem"
local KEY_VENDOR_SPELL="vndspl"
local KEY_PUBLIC_EVENT="pe"
local KEY_CHALLENGE="chg"
local KEY_CHALLENGE_REWARDS="chgrw"
local KEY_PATH_EPISODE="pathe"
local KEY_PATH_MISSION="pathm"
local KEY_TRADESKILL="trades"
local KEY_TRADESKILL_TALENT="tradet"
local KEY_SCHEMATIC="sch"
local KEY_DATACUBE="dc"
local KEY_ATTRIBUTE="attr"
local KEY_FACTION="rep"
local KEY_HOUSE_PLOTS="plot"
local KEY_HOUSE_DECORTYPE="decort"
local KEY_HOUSE_DECOR="decor"
local KEY_ACHI_GROUP="achg"
local KEY_ACHI="ach"
local KEY_CHARACTERS="characters"
local KEY_CONTRACTS="contracts"
local KEY_CONTRACTREWARDS="contractrewards"

-----------------------------------------------------------------------------------------------
-- Initialization
-----------------------------------------------------------------------------------------------
function Jabbithole:new(o)
    o = o or {}
    setmetatable(o, self)
    self.__index = self 

	self.tSavedData={}
	self.tSavedData["v"]=0
	self.accountVars={}

    -- initialize variables here

    return o
end

function Jabbithole:Init()
    Apollo.RegisterAddon(self)
end

function Jabbithole:OnLoad()
	self.xmlDoc = XmlDoc.CreateFromFile("Jabbithole.xml")
	Apollo.LoadSprites("JabbitholeSprites.xml", "JabbitholeSprites")
	self.xmlDoc:RegisterCallback("OnDocumentReady", self)
end 

function Jabbithole:OnDocumentReady()

	if GameLib.GetPlayerUnit() then
		self:OnCharacterCreated()
	end

	Apollo.RegisterEventHandler("InterfaceMenuListHasLoaded","OnInterfaceMenuListHasLoaded", self)
	Apollo.RegisterEventHandler("WindowManagementReady","OnWindowManagementReady", self)
	
	Apollo.RegisterTimerHandler("DelayedInitTimer", "OnDelayedInitTimer", self)
	Apollo.CreateTimer("DelayedInitTimer", 0.1, false)
	--Apollo.StopTimer("DelayedInitTimer")
	
	Apollo.RegisterEventHandler("CharacterCreated","OnCharacterCreated", self)
	Apollo.RegisterEventHandler("Dialog_ShowState", "OnDialog_ShowState", self)
	Apollo.RegisterEventHandler("VarChange_ZoneName","OnSubZoneChanged", self)
	Apollo.RegisterEventHandler("SubZoneChanged","OnSubZoneChanged", self)
	Apollo.RegisterEventHandler("ChangeWorld","OnWorldChanged", self)
	Apollo.RegisterEventHandler("PlayerPathMissionUnlocked","OnSubZoneChanged", self)
	Apollo.RegisterEventHandler("UnitCreated", "OnUnitCreated", self)
	Apollo.RegisterEventHandler("UnitDestroyed", "OnUnitDestroyed", self)
	Apollo.RegisterEventHandler("LootedItem","OnLootedItem", self)
	Apollo.RegisterEventHandler("PlayerTitleUpdate","OnPlayerTitleUpdate", self)
	Apollo.RegisterEventHandler("TargetUnitChanged","OnTargetUnitChanged", self)
	Apollo.RegisterEventHandler("CombatLogDamage","OnCombatLogDamage", self)
	Apollo.RegisterEventHandler("VendorItemsUpdated","OnVendorItemsUpdated", self)
	Apollo.RegisterEventHandler("InvokeVendorWindow","OnInvokeVendorWindow", self)
	Apollo.RegisterEventHandler("ItemModified", "OnItemModified", self)
	Apollo.RegisterEventHandler("PublicEventStart", "OnPublicEventStart", self)
	Apollo.RegisterEventHandler("ChallengeActivate", "OnChallengeActivate", self)
--	Apollo.RegisterEventHandler("ChallengeRewardListReady", "OnChallengeRewardListReady", self)
	Apollo.RegisterEventHandler("ItemRemoved","OnItemRemoved", self)
	Apollo.RegisterEventHandler("CraftingSchematicLearned", "OnCraftingSchematicLearned", self)
	Apollo.RegisterEventHandler("DatacubeUpdated", "OnDatacubeUpdated", self)
	Apollo.RegisterEventHandler("PublicEventInitiateVote", 	"OnPublicEventInitiateVote", self)
	Apollo.RegisterEventHandler("PlayerPathMissionUnlocked", "OnPlayerPathMissionUnlocked", self)
	Apollo.RegisterEventHandler("PlayerPathMissionDeactivate", "OnPlayerPathMissionDeactivate", self)
	Apollo.RegisterEventHandler("HousingVendorListRecieved", "OnHousingPlugItemsUpdated", self)
	Apollo.RegisterEventHandler("ItemSentToCrate", "OnItemSentToCrate", self)
	Apollo.RegisterEventHandler("GenericEvent_PlayerCampStart", "OnPlayerExit", self)
	Apollo.RegisterEventHandler("GenericEvent_PlayerExitStart", "OnPlayerExit", self)
	Apollo.RegisterEventHandler("DyeLearned", "OnDyeLearned", self)
	Apollo.RegisterEventHandler("ContractBoardOpen", "OnContractBoardOpen", self)
		
	Apollo.RegisterEventHandler("ToggleJabbitholeWindow", "OnJabbitholeOn", self)
	
	Apollo.RegisterSlashCommand("jh", "OnJabbitholeOn", self)

	self.wndMain = Apollo.LoadForm(self.xmlDoc, "JabbitholeForm", nil, self)
	self.wndMain:Show(false)
	if self.accountVars.warnUpload == nil or self.accountVars.warnUpload then
		self.wndMain:FindChild("WarnButton"):SetCheck(true)
	end
	if self.accountVars.saveArmory then
		self.wndMain:FindChild("ArmoryButton"):SetCheck(true)
	end
	
	self.wndWarning = Apollo.LoadForm(self.xmlDoc, "JabbitholeWarningForm", nil, self)
	self.wndWarning:Show(false)
	self.wndExport = Apollo.LoadForm(self.xmlDoc, "CharacterExportForm", nil, self)
	self.wndExport:Show(false)
	
end

function Jabbithole:OnInterfaceMenuListHasLoaded()
	Event_FireGenericEvent("InterfaceMenuList_NewAddOn", "Jabbithole", {"ToggleJabbitholeWindow", "", "JabbitholeSprites:MenuLogo"})
end

function Jabbithole:OnWindowManagementReady()
	Event_FireGenericEvent("WindowManagementAdd", {wnd = self.wndMain, strName = 'Jabbithole'})
end

function Jabbithole:OnDelayedInitTimer()
--	Print("Jabbithole:OnDelayedInitTimer()")
	if not self.bInitializedProperly then
--		Print("Jabbithole is loading!")
		self:SetupInternals()
		self.bInitializedProperly = (GameLib.IsCharacterLoaded() and self.tZone and self.tPlayer and self.tPlayer:IsValid())
		Apollo.StartTimer("DelayedInitTimer")
	else
		Apollo.StopTimer("DelayedInitTimer")
		ChatSystemLib.PostOnChannel(ChatSystemLib.ChatChannel_System, "Jabbithole is loaded, thanks for contributing!", "")
		
		self.knownDyes=GameLib.GetKnownDyes()
		
--		Print("Jabbithole is now loaded, thanks for contributing!")

		local v=GameLib.GetVersionInfo()
		if v then
			self.tSavedData["build"]=v.nBuildNumber
		end

		self:AddSpell(GameLib.GetSpell(46867))
		self:AddSpell(GameLib.GetSpell(47022))
		self:AddSpell(GameLib.GetSpell(47866))
		self:AddSpell(GameLib.GetSpell(47881))
		self:AddSpell(GameLib.GetSpell(58586))
		self:AddSpell(GameLib.GetSpell(69450))
		self:AddSpell(GameLib.GetSpell(46073))
		self:AddSpell(GameLib.GetSpell(38782))
		self:AddSpell(GameLib.GetSpell(46074))
		self:AddSpell(GameLib.GetSpell(47273))
		
		self:OnSubZoneChanged()
		self:OnPlayerTitleUpdate()
		self:AddTradeskills()
		self:AddReputations()
		self:SaveHousingStuff()
		self:OnContractBoardOpen()
		
		if lastUploadWarning+79200<os.time() then
			if self.NumItems then
				if self.NumItems>3500 or self.NumSchematics>1500 or self.NumSpells>1500 or self.NumCreatures>1000 or self.NumQuests>500 then
					if self.accountVars.warnUpload == true or self.accountVars.warnUpload == nil then
						self.wndWarning:Show(true)
					end
				end
			end
		end
		self:AddCharacter()
	end
end

function Jabbithole:AddItemProfRequirement(data)
	if data then
		if self.tSavedData[KEY_ITEM_PROF_REQ] == nil then
			self.tSavedData[KEY_ITEM_PROF_REQ]={}
		end
		if self.tSavedData[KEY_ITEM_PROF_REQ][data.nId] ~= data.strName then
			self.tSavedData[KEY_ITEM_PROF_REQ][data.nId]=data.strName
		end
		return data.nId
	end
	return -1
end

function Jabbithole:AddItemAttribute(id)
	if self.tSavedData[KEY_ATTRIBUTE] == nil then
		self.tSavedData[KEY_ATTRIBUTE]={}
	end
	local name = Item.GetPropertyName(id)
	if name then
		if self.tSavedData[KEY_ATTRIBUTE][id] ~= name then
			self.tSavedData[KEY_ATTRIBUTE][id]=name
		end
	end
end

function Jabbithole:AddItemSlot(id,name)
	if id and name then
		if self.tSavedData[KEY_ITEM_SLOT] == nil then
			self.tSavedData[KEY_ITEM_SLOT]={}
		end
		if self.tSavedData[KEY_ITEM_SLOT][id] ~= name then
			self.tSavedData[KEY_ITEM_SLOT][id]=name
		end
		return id
	end
	return -1
end

function Jabbithole:AddItemType(id,name)
	if id and name then
		if self.tSavedData[KEY_ITEM_TYPE] == nil then
			self.tSavedData[KEY_ITEM_TYPE]={}
		end
		if self.tSavedData[KEY_ITEM_TYPE][id] ~= name then
			self.tSavedData[KEY_ITEM_TYPE][id]=name
		end
		return id
	end
	return -1
end

function Jabbithole:AddItemFamily(id,name)
	if id and name then
		if self.tSavedData[KEY_ITEM_FAMILY] == nil then
			self.tSavedData[KEY_ITEM_FAMILY]={}
		end
		if self.tSavedData[KEY_ITEM_FAMILY][id] ~= name then
			self.tSavedData[KEY_ITEM_FAMILY][id]=name
		end
		return id
	end
	return -1
end

function Jabbithole:AddItemCategory(id,name)
	if id and name then
		if self.tSavedData[KEY_ITEM_CATEGORY] == nil then
			self.tSavedData[KEY_ITEM_CATEGORY]={}
		end
		if self.tSavedData[KEY_ITEM_CATEGORY][id] ~= name then
			self.tSavedData[KEY_ITEM_CATEGORY][id]=name
		end
		return id
	end
	return -1
end

function Jabbithole:AddVendorItem(unit,item)
	if unit and unit:IsValid() and item then
		if self.tSavedData[KEY_VENDOR_ITEM] == nil then
			self.tSavedData[KEY_VENDOR_ITEM]={}
		end
		local vid=self:MakeCreatureId(unit)

		if self.tSavedData[KEY_VENDOR_ITEM][vid] == nil then
			self.tSavedData[KEY_VENDOR_ITEM][vid]={}
		end
		local key = self:MakeItemId(item.itemData)
		if self.tSavedData[KEY_VENDOR_ITEM][vid][key] == nil then
			self.tSavedData[KEY_VENDOR_ITEM][vid][key]={}
		end
		local specitem="-"
		if item.bIsSpecial then
			specitem="+"
		end
		local ct1="-1"
		local ct2="-1"
		if item.tPriceInfo.eCurrencyType1 then
			ct1=item.tPriceInfo.eCurrencyType1
		else
			if item.tPriceInfo.itemExchange1 then
				ct1="i"..item.tPriceInfo.itemExchange1:GetItemId()
			end
		end
		if item.tPriceInfo.eCurrencyType2 then
			ct2=item.tPriceInfo.eCurrencyType2
		else
			if item.tPriceInfo.itemExchange2 then
				ct2="i"..item.tPriceInfo.itemExchange2:GetItemId()
			end
		end
		self.tSavedData[KEY_VENDOR_ITEM][vid][key]=specitem.."/"..item.nStockCount.."/"..item.nStackSize.."/"..item.tPriceInfo.nAmount1.."/"..ct1.."/"..item.tPriceInfo.nAmount2.."/"..ct2
		if item.idPrereq > 0 then
			local tPrereqInfo = GameLib.GetPlayerUnit():GetPrereqInfo(item.idPrereq)
			if tPrereqInfo and tPrereqInfo.strText then
				self.tSavedData[KEY_VENDOR_ITEM][vid][key]=self.tSavedData[KEY_VENDOR_ITEM][vid][key].."/"..tPrereqInfo.strText
			end
		end			
	end
end

function Jabbithole:AddVendorSpell(unit,item)
	if unit and unit:IsValid() and item then
		if self.tSavedData[KEY_VENDOR_SPELL] == nil then
			self.tSavedData[KEY_VENDOR_SPELL]={}
		end
		local vid=self:MakeCreatureId(unit)

		if self.tSavedData[KEY_VENDOR_SPELL][vid] == nil then
			self.tSavedData[KEY_VENDOR_SPELL][vid]={}
		end
		self:AddSpell(item.splData)
		local key = item.splData:GetId()
		if self.tSavedData[KEY_VENDOR_SPELL][vid][key] == nil then
			self.tSavedData[KEY_VENDOR_SPELL][vid][key]={}
		end
		local specitem="-"
		if item.bIsSpecial then
			specitem="+"
		end
		self.tSavedData[KEY_VENDOR_SPELL][vid][key]=specitem.."/"..item.nStockCount.."/"..item.nStackSize.."/"..item.tPriceInfo.nAmount1.."/"..item.tPriceInfo.eCurrencyType1.."/"..item.tPriceInfo.nAmount2.."/"..item.tPriceInfo.eCurrencyType2
		if item.idPrereq > 0 then
			local tPrereqInfo = GameLib.GetPlayerUnit():GetPrereqInfo(item.idPrereq)
			if tPrereqInfo and tPrereqInfo.strText then
				self.tSavedData[KEY_VENDOR_SPELL][vid][key]=self.tSavedData[KEY_VENDOR_SPELL][vid][key].."/"..tPrereqInfo.strText
			end
		end			
	end
end

function Jabbithole:MakeItemId(item)
	local id="invalid"
	if item then
		id=item:GetItemId()
--[[		id=item:GetChatLinkString()
		local info=item:GetDetailedInfo().tPrimary
		local rnp = 0
		if info.arRandomProperties then
			rnp = #info.arRandomProperties
		end
		local smin = 0
		local smax = 0
		if info.tRunes then
			if info.tRunes.nMaximum then
				smax=info.tRunes.nMaximum 
			end
			if info.tRunes.nMinimum then
				smin=info.tRunes.nMinimum 
			end
		end
		id = id.."-"..rnp.."-"..smin.."-"..smax]]
	end
	return id
end

function Jabbithole:AddItem(item, questId, unit, bSaveSide, bIsRoot)
	if item then
		if self.tSavedData[KEY_ITEMS] == nil then
			self.tSavedData[KEY_ITEMS]={}
		end
		local id=self:MakeItemId(item)

		if self.tSavedData[KEY_ITEMS][id] == nil then
			self.tSavedData[KEY_ITEMS][id]={}
		else
			if bIsRoot then
				if rootProcessed[id] then
					return id
				else
					rootProcessed[id] = true
				end
			end
		end

		if bIsRoot then
			self.tSavedData[KEY_ITEMS][id]["root"]=bIsRoot
		end
		
		local info=item:GetDetailedInfo().tPrimary

		if item:GetGivenQuest() then
			local q=item:GetGivenQuest()
			self:AddQuest(q,nil,nil, bSaveSide, bIsRoot)
			self.tSavedData[KEY_ITEMS][id]["quest"]=q:GetId()
		end
		
		self.tSavedData[KEY_ITEMS][id]["name"]=info.strName
		self.tSavedData[KEY_ITEMS][id]["id"]=item:GetItemId()
		self.tSavedData[KEY_ITEMS][id]["cls"]=item:GetChatLinkString()
		
		self.tSavedData[KEY_ITEMS][id]["icon"]=item:GetIcon()
		
		if self.tSavedData[KEY_ITEMS][id]["drops"]==nil then
			self.tSavedData[KEY_ITEMS][id]["drops"]={}
		end
		
		self.tSavedData[KEY_ITEMS][id]["binds"]=0
		if info.tBind then
			if info.tBind.bOnEquip then
				self.tSavedData[KEY_ITEMS][id]["binds"]=1
			end
			if info.tBind.bOnPickup then
				self.tSavedData[KEY_ITEMS][id]["binds"]=2
			end
		end
		
		self.tSavedData[KEY_ITEMS][id]["salv"]=(info.bSalvagable==true)
		self.tSavedData[KEY_ITEMS][id]["q"]=info.eQuality
		self.tSavedData[KEY_ITEMS][id]["pwr"]=item:GetItemPower()
		self.tSavedData[KEY_ITEMS][id]["flav"]=info.strFlavor

		if info.tStack then
			self.tSavedData[KEY_ITEMS][id]["stck"]=info.tStack.nMaxCount
		end
		if info.tCharge then
			if info.tCharge.nMaxCount then
				self.tSavedData[KEY_ITEMS][id]["chgs"]=info.tCharge.nMaxCount
			end
		end
		if info.tLevelRequirement then
			self.tSavedData[KEY_ITEMS][id]["rqlvl"]=info.tLevelRequirement.nLevelRequired
		else
			self.tSavedData[KEY_ITEMS][id]["rqlvl"]=0
		end
		if info.tProfRequirement then
			self.tSavedData[KEY_ITEMS][id]["rqprof"]=self:AddItemProfRequirement(info.tProfRequirement)				
		else
			self.tSavedData[KEY_ITEMS][id]["rqprof"]=0
		end
		self.tSavedData[KEY_ITEMS][id]["uniq"]=item:IsUnique()
			
		if info.tUnique and info.tUnique.bEquipped then
			self.tSavedData[KEY_ITEMS][id]["uniqeq"]=true
		end
		
		self.tSavedData[KEY_ITEMS][id]["equip"]=item:IsEquippable()
		self.tSavedData[KEY_ITEMS][id]["comm"]=item:IsCommodity()
		self.tSavedData[KEY_ITEMS][id]["data"]=item:isData()
		self.tSavedData[KEY_ITEMS][id]["inst"]=item:isInstance()
		self.tSavedData[KEY_ITEMS][id]["tssid"]=item:GetTradeskillSchematicId()
		
		--self.tSavedData[KEY_ITEMS][id]["wpdmgmx"]=item:GetWeaponDamageMax() removed in reloaded
		--self.tSavedData[KEY_ITEMS][id]["wpdmgmn"]=item:GetWeaponDamageMin() removed in reloaded
		--self.tSavedData[KEY_ITEMS][id]["wppw"]=item:GetWeaponPower() removed in reloaded
		--self.tSavedData[KEY_ITEMS][id]["wpspd"]=item:GetWeaponSpeed() removed in reloaded
		self.tSavedData[KEY_ITEMS][id]["blevel"]=item:GetPowerLevel()
		self.tSavedData[KEY_ITEMS][id]["deco"]=item:GetHousingDecorInfoId()
		if info.nEffectiveLevel then
			self.tSavedData[KEY_ITEMS][id]["elevel"]=info.nEffectiveLevel
		end

		self.tSavedData[KEY_ITEMS][id]["cat"]=self:AddItemCategory(info.eCategory, item:GetItemCategoryName())
		self.tSavedData[KEY_ITEMS][id]["fam"]=self:AddItemFamily(info.eFamily, item:GetItemFamilyName())
		self.tSavedData[KEY_ITEMS][id]["typ"]=self:AddItemType(info.eType, item:GetItemTypeName())
		if item:GetSlot() then
			self.tSavedData[KEY_ITEMS][id]["slot"]=self:AddItemSlot(item:GetSlot(), item:GetSlotName())
		else
			self.tSavedData[KEY_ITEMS][id]["slot"]=-1
		end

		if info.arSpells then
			self.tSavedData[KEY_ITEMS][id]["spls"]={}
			for idx = 1, #info.arSpells do
				self.tSavedData[KEY_ITEMS][id]["spls"][idx]={}
				self.tSavedData[KEY_ITEMS][id]["spls"][idx]["id"]=self:AddSpell(info.arSpells[idx].splData)
				self.tSavedData[KEY_ITEMS][id]["spls"][idx]["name"]=info.arSpells[idx].strName
				if info.arSpells[idx].bActivate == true then
					self.tSavedData[KEY_ITEMS][id]["spls"][idx]["use"]=1
				end
				if info.arSpells[idx].bOnEquip== true then
					self.tSavedData[KEY_ITEMS][id]["spls"][idx]["equ"]=1
				end
				if info.arSpells[idx].bProc == true then
					self.tSavedData[KEY_ITEMS][id]["spls"][idx]["prc"]=1
				end
			end
		end
					
		--sold for item:GetDetailedInfo
		--Money.CodeEnumCurrencyType
		if info.tCost and info.tCost.arMonSell then
			self.tSavedData[KEY_ITEMS][id]["sold4"]=info.tCost.arMonSell[1]:GetAmount()
			self.tSavedData[KEY_ITEMS][id]["soldc"]=info.tCost.arMonSell[1]:GetMoneyType()
		end

		--imbuements
		if info.arImbuements then
			self.tSavedData[KEY_ITEMS][id]["imbue2"]={}
			for key, imb in pairs(info.arImbuements) do
				if imb.queImbuement then
					self:AddQuest(imb.queImbuement,nil,nil,bSaveSide,bIsRoot)
					self.tSavedData[KEY_ITEMS][id]["imbue2"][key]={}
					self.tSavedData[KEY_ITEMS][id]["imbue2"][key]["i"]=imb.queImbuement:GetId()
					self.tSavedData[KEY_ITEMS][id]["imbue2"][key]["n"]=imb.strName
					self.tSavedData[KEY_ITEMS][id]["imbue2"][key]["o"]=imb.strObjective
					self.tSavedData[KEY_ITEMS][id]["imbue2"][key]["s"]=imb.strSpecial
					self.tSavedData[KEY_ITEMS][id]["imbue2"][key]["a"]=imb.bActive
					self.tSavedData[KEY_ITEMS][id]["imbue2"][key]["c"]=imb.bComplete
					self.tSavedData[KEY_ITEMS][id]["imbue2"][key]["at"]=imb.eProperty
					self.tSavedData[KEY_ITEMS][id]["imbue2"][key]["v"]=imb.nValue
				end
			end				
		end
					
		--primary innate stats item:GetDetailedInfo
		if info.arInnateProperties then
			self.tSavedData[KEY_ITEMS][id]["stati"]={}
			for key, prop in pairs(info.arInnateProperties) do
				self.tSavedData[KEY_ITEMS][id]["stati"][prop.eProperty.."/"..prop.nSortOrder.."/"..prop.nValue]=1
				self:AddItemAttribute(prop.eProperty)
			end
		end
		
		--primary budget stats item:GetDetailedInfo
		--primary budget derived item:GetDetailedInfo
		self.tSavedData[KEY_ITEMS][id]["statvar"]={}
		local statsKey=""
		if info.arBudgetBasedProperties then
			for idx, prop in pairs(info.arBudgetBasedProperties) do
				if idx>1 then
					statsKey = statsKey.."#"
				end
				statsKey = statsKey..prop.eProperty.."/"..prop.nSortOrder.."/"..prop.nValue
				self:AddItemAttribute(prop.eProperty)
			end
		end
		statsKey = statsKey .. "%"			
		--sigils, name,type item:GetDetailedInfo + item:GetSigils
		if info.tRunes then
			if info.tRunes.arRuneSlots then
				for idx = 1, #info.tRunes.arRuneSlots do
					if idx>1 then
						statsKey = statsKey.."#"
					end
					statsKey = statsKey..info.tRunes.arRuneSlots[idx].eElement
				end
				if info.tRunes.arRuneSlots[1] and info.tRunes.arRuneSlots[1].nBudget then
					self.tSavedData[KEY_ITEMS][id]["sigval"]=info.tRunes.arRuneSlots[1].nBudget
				end
			end
			if info.tRunes.nMaximum then
				self.tSavedData[KEY_ITEMS][id]["sigmax"]=info.tRunes.nMaximum 
			end
			if info.tRunes.nMinimum then
				self.tSavedData[KEY_ITEMS][id]["sigmin"]=info.tRunes.nMinimum 
			end
		end
		if statsKey ~= "" and statsKey ~= "%" then
			self.tSavedData[KEY_ITEMS][id]["statvar"][statsKey]=1
		end

--[[ changed in reloaded
		if info.arRandomProperties then
			self.tSavedData[KEY_ITEMS][id]["statrc"]=#info.arRandomProperties
			
			self.tSavedData[KEY_ITEMS][id]["rndprops"]={}
			for idx, rp in pairs(info.arRandomProperties or {}) do
				self.tSavedData[KEY_ITEMS][id]["rndprops"][idx]=rp.strName.."//"
				if rp.itemRandom then
					self.tSavedData[KEY_ITEMS][id]["rndprops"][idx]=self.tSavedData[KEY_ITEMS][id]["rndprops"][idx]..self:AddItem(rp.itemRandom, -1, nil, false, bIsRoot)
				end
			end
		end
]]

		if info.tUndefinedProperties then
			self.tSavedData[KEY_ITEMS][id]["statrc"]=info.tUndefinedProperties.nRandomProperties
			self.tSavedData[KEY_ITEMS][id]["statcc"]=info.tUndefinedProperties.nCraftableProperties
		end
		
		self.tSavedData[KEY_ITEMS][id]["runepve"] = info.bPveOnlyRune
		self.tSavedData[KEY_ITEMS][id]["runeminilvl"] = info.nInstalledMinimumItemLevel
		if info.tRuneSet then
			self.tSavedData[KEY_ITEMS][id]["runeset"]={}
			self.tSavedData[KEY_ITEMS][id]["runeset"]["n"]=info.tRuneSet.strName
			self.tSavedData[KEY_ITEMS][id]["runeset"]["mp"]=info.tRuneSet.nMaxPower
			self.tSavedData[KEY_ITEMS][id]["runeset"]["np"]=info.tRuneSet.nPower
			self.tSavedData[KEY_ITEMS][id]["runeset"]["id"]=info.tRuneSet.nSetId
			if info.tRuneSet.arBonuses then
				self.tSavedData[KEY_ITEMS][id]["runeset"]["bs"]={}
				for idx, bo in pairs(info.tRuneSet.arBonuses or {}) do
					self.tSavedData[KEY_ITEMS][id]["runeset"]["bs"][idx]={}
					self.tSavedData[KEY_ITEMS][id]["runeset"]["bs"][idx]["def"]=bo.bDefault
					self.tSavedData[KEY_ITEMS][id]["runeset"]["bs"][idx]["prop"]=bo.eProperty
					self.tSavedData[KEY_ITEMS][id]["runeset"]["bs"][idx]["pow"]=bo.nPower
					self.tSavedData[KEY_ITEMS][id]["runeset"]["bs"][idx]["sca"]=bo.nScalar
					self.tSavedData[KEY_ITEMS][id]["runeset"]["bs"][idx]["so"]=bo.nSortOrder
					self.tSavedData[KEY_ITEMS][id]["runeset"]["bs"][idx]["val"]=bo.nValue
					self.tSavedData[KEY_ITEMS][id]["runeset"]["bs"][idx]["n"]=bo.strName
					self.tSavedData[KEY_ITEMS][id]["runeset"]["bs"][idx]["act"]=bo.bActivate
					self.tSavedData[KEY_ITEMS][id]["runeset"]["bs"][idx]["equ"]=bo.bOnEquip
					if bo.splData then
						self.tSavedData[KEY_ITEMS][id]["runeset"]["bs"][idx]["spell"]=self:AddSpell(bo.splData)
					end
				end
			end
		end
--[[ removed in reloaded
		local mc=item:GetMicrochipInfo()
		if mc then
			self.tSavedData[KEY_ITEMS][id]["microcinfo"]={}
			self.tSavedData[KEY_ITEMS][id]["microcinfo"]["etype"]=mc.eType
			self.tSavedData[KEY_ITEMS][id]["microcinfo"]["uprop"]=mc.idUnitProperty
			self:AddItemAttribute(mc.idUnitProperty)
			self.tSavedData[KEY_ITEMS][id]["microcinfo"]["gname"]=mc.strRandomCircuitGroupName
			self.tSavedData[KEY_ITEMS][id]["microcinfo"]["gw"]=mc.fRandomCircuitGroupTotalWeight
			if mc.arRandomCircuits then
				self.tSavedData[KEY_ITEMS][id]["microcinfo"]["rndcs"]={}
				for idx, mcc in pairs(mc.arRandomCircuits or {}) do
					self.tSavedData[KEY_ITEMS][id]["microcinfo"]["rndcs"][idx]=mcc.fWeight.."//"
					if mcc.itemCircuit then
						self.tSavedData[KEY_ITEMS][id]["microcinfo"]["rndcs"][idx]=self.tSavedData[KEY_ITEMS][id]["microcinfo"]["rndcs"][idx]..self:AddItem(mcc.itemCircuit, -1, nil, false, bIsRoot)
					end
				end
			end
			if mc.tSet then
				self.tSavedData[KEY_ITEMS][id]["microcinfo"]["set"]={}
				for idx, st in pairs(mc.tSet or {}) do
					self.tSavedData[KEY_ITEMS][id]["microcinfo"]["set"][idx]={}
					self.tSavedData[KEY_ITEMS][id]["microcinfo"]["set"][idx]["n"]=st.strName
					self.tSavedData[KEY_ITEMS][id]["microcinfo"]["set"][idx]["mp"]=st.nMaxPower
					self.tSavedData[KEY_ITEMS][id]["microcinfo"]["set"][idx]["np"]=st.nPower
					self.tSavedData[KEY_ITEMS][id]["microcinfo"]["set"][idx]["b"]={}
					for idx2, bo in pairs(st.arBonuses or {}) do
						self.tSavedData[KEY_ITEMS][id]["microcinfo"]["set"][idx]["b"][idx2]=bo.nPower.."//"
						if bo.splBonus then
							self.tSavedData[KEY_ITEMS][id]["microcinfo"]["set"][idx]["b"][idx2]=self.tSavedData[KEY_ITEMS][id]["microcinfo"]["set"][idx]["b"][idx2]..self:AddSpell(bo.splBonus)
						end
					end
				end
			end
		end
		
		if info.tChipInfo then
			self.tSavedData[KEY_ITEMS][id]["chipinfo"]=info.tChipInfo.eType.."//"
			if info.tChipInfo.eProperty then
				self.tSavedData[KEY_ITEMS][id]["chipinfo"]=self.tSavedData[KEY_ITEMS][id]["chipinfo"]..info.tChipInfo.eProperty
				self:AddItemAttribute(info.tChipInfo.eProperty)
			end
			self.tSavedData[KEY_ITEMS][id]["chipspells"]={}
			for idx, cis in pairs(info.tChipInfo.arSpells or {}) do
				self.tSavedData[KEY_ITEMS][id]["chipspells"][idx]={}
				self.tSavedData[KEY_ITEMS][id]["chipspells"][idx]["act"]=cis.bActivate
				self.tSavedData[KEY_ITEMS][id]["chipspells"][idx]["equ"]=cis.bOnEquip
				self.tSavedData[KEY_ITEMS][id]["chipspells"][idx]["pro"]=cis.bProc
				self.tSavedData[KEY_ITEMS][id]["chipspells"][idx]["flav"]=cis.strFlavor
				self.tSavedData[KEY_ITEMS][id]["chipspells"][idx]["name"]=cis.strName
				self.tSavedData[KEY_ITEMS][id]["chipspells"][idx]["spl"]=self:AddSpell(cis.splData)
			end
		end
]]
					
		-- item:GetRequiredClass {} NEEDS DATA
		--race req item:GetRequiredRace - nil NEEDS DATA
		
		if info.tDurability then
			self.tSavedData[KEY_ITEMS][id]["dura"]=info.tDurability.nMax
			if info.tDurability.nCurrent == nil then
				-- no current durability most likely means it is in vendor inventory
				-- info.arRandomProperties shall list the random properties
				-- item:isInstance() shall be false
				-- might be useful for something
			end
		end
		
		self.tSavedData[KEY_ITEMS][id]["tsreq"]={}
		for idx, ts in pairs(info.arTradeskillReqs or {}) do
			self.tSavedData[KEY_ITEMS][id]["tsreq"][ts.strName]=ts.eTier
		end
		self.tSavedData[KEY_ITEMS][id]["classreq"]=""
		for idx, cls in pairs(item:GetRequiredClass() or {}) do
			if idx>1 then
				self.tSavedData[KEY_ITEMS][id]["classreq"]=self.tSavedData[KEY_ITEMS][id]["classreq"]..","
			end
			self.tSavedData[KEY_ITEMS][id]["classreq"]=self.tSavedData[KEY_ITEMS][id]["classreq"]..cls.idClassReq
		end
		
	
		if unit then
			local unitid = self:MakeCreatureId(unit)

			if self.tSavedData[KEY_ITEMS][id]["drops"][unitid] == nil then
				self.tSavedData[KEY_ITEMS][id]["drops"][unitid]=1
			else
				self.tSavedData[KEY_ITEMS][id]["drops"][unitid]=self.tSavedData[KEY_ITEMS][id]["drops"][unitid]+1
			end
		end
				
		return id
	end
end

function Jabbithole:AddSpell(splSource)
	if splSource then
		if self.tSavedData[KEY_SPELLS] == nil then
			self.tSavedData[KEY_SPELLS]={}
		end
		local id=splSource:GetId()
		if self.tSavedData[KEY_SPELLS][id] == nil then
			self.tSavedData[KEY_SPELLS][id]={}
			self.tSavedData[KEY_SPELLS][id]["name"]=splSource:GetName()
			self.tSavedData[KEY_SPELLS][id]["flav"]=nlFix(splSource:GetFlavor())
			self.tSavedData[KEY_SPELLS][id]["bid"]=splSource:GetBaseSpellId()
			self.tSavedData[KEY_SPELLS][id]["gcd"]=splSource:GetGCDTime()

			local fCastTime
			local strCastInfo = splSource:GetCastInfoString()
			local eCastMethod = splSource:GetCastMethod()
			local tChannelData = splSource:GetChannelData()

			if strCastInfo and strCastInfo ~= "" then
				fCastTime = splSource:GetCastTimeOverride()
			else
				if eCastMethod == Spell.CodeEnumCastMethod.Channeled or eCastMethod == Spell.CodeEnumCastMethod.ChanneledField then
					fCastTime = tChannelData["fMaxTime"]
				elseif eCastMethod == Spell.CodeEnumCastMethod.PressHold or eCastMethod == Spell.CodeEnumCastMethod.ChargeRelease then
					fCastTime = splSource:GetThresholdTime()
				else
					fCastTime = splSource:GetCastTime()
				end
		
				if eCastMethod == Spell.CodeEnumCastMethod.Normal or eCastMethod == Spell.CodeEnumCastMethod.Multiphase or eCastMethod == Spell.CodeEnumCastMethod.Aura then
					if fCastTime == 0 then
						strCastInfo = Apollo.GetString("Tooltip_Instant")
					else
						strCastInfo = String_GetWeaselString(Apollo.GetString("Tooltip_CastTime"), tostring(strRound(fCastTime, 2)))
					end
				elseif eCastMethod == Spell.CodeEnumCastMethod.Channeled or eCastMethod == Spell.CodeEnumCastMethod.ChanneledField then
					strCastInfo = String_GetWeaselString(Apollo.GetString("Tooltip_ChannelTime"), tostring(strRound(fCastTime, 2)))
				elseif eCastMethod == Spell.CodeEnumCastMethod.PressHold then
					strCastInfo = String_GetWeaselString(Apollo.GetString("Tooltip_HoldTime"), tostring(strRound(fCastTime, 2)))
				elseif eCastMethod == Spell.CodeEnumCastMethod.ClientSideInteraction then
					strCastInfo = Apollo.GetString("Tooltip_CSI")
				elseif eCastMethod == Spell.CodeEnumCastMethod.RapidTap then
					if fCastTime == 0 then
						strCastInfo = Apollo.GetString("Tooltips_InstantMultiTap")
					else
						strCastInfo = String_GetWeaselString(Apollo.GetString("Tooltips_CastThenMultiTap"), tostring(strRound(fCastTime, 2)))
					end
				elseif eCastMethod == Spell.CodeEnumCastMethod.ChargeRelease then
					strCastInfo = String_GetWeaselString(Apollo.GetString("Tooltips_ChargeTime"), tostring(strRound(fCastTime, 2)))
				else
					strCastInfo = Apollo.GetString("Tooltips_UnknownCastMethod")
				end
			end
			
			self.tSavedData[KEY_SPELLS][id]["castinfo"]=strCastInfo 
			
			-- Range
			
			self.tSavedData[KEY_SPELLS][id]["rmin"]=splSource:GetMinimumRange()
			self.tSavedData[KEY_SPELLS][id]["rmax"]=splSource:GetMaximumRange()
	
			local strCost = splSource:GetCostInfoString() -- always use the override string if available
			if not strCost or strCost == "" then
				local tResource = {}
				local tCosts = splSource:GetCasterInnateCosts()
		
				if not tCosts or #tCosts == 0 then
					tCosts = splSource:GetCasterInnateRequirements()
				end
		
				for idx = 1, #tCosts do
					if strCost ~= "" then
						strCost = strCost .. Apollo.GetString("Tooltips_And")
					end
		
					tResource = tVitals[tCosts[idx].eVital]
					if not tResource then
						tResource = {}
						tResource.strName = String_GetWeaselString(Apollo.GetString("Tooltips_UnknownVital"), tCosts[idx].eVital)
					end
		
					tResourceData =
					{
						["name"]	= tResource.strName,
						["count"]	= tCosts[idx].nValue,
					}
					strCost = String_GetWeaselString(Apollo.GetString("Tooltips_SpellCost"), strCost, tResourceData)
		
					if eCastMethod == Spell.CodeEnumCastMethod.Channeled then
						strCost = String_GetWeaselString(Apollo.GetString("Tooltips_ChanneledCost"), strCost)
					elseif eCastMethod == Spell.CodeEnumCastMethod.ChargeRelease then
						strCost = String_GetWeaselString(Apollo.GetString("Tooltips_Charges"), strCost)
					end
				end
			end
						
			self.tSavedData[KEY_SPELLS][id]["castcost"]=strCost 
		
			-- Targeting
			if splSource:IsFreeformTarget() then
				self.tSavedData[KEY_SPELLS][id]["target"]=Apollo.GetString("Tooltips_Freeform")
			elseif splSource:IsSelfSpell() then
				self.tSavedData[KEY_SPELLS][id]["target"]=Apollo.GetString("Tooltips_Self")
			else
				self.tSavedData[KEY_SPELLS][id]["target"]=Apollo.GetString("Tooltips_Targeted")
			end
							
			-- Cooldown / Recharge
		    local fCooldownTime = splSource:GetCooldownTime()
		    local tCharges = splSource:GetAbilityCharges()
		
			if splSource:ShouldHideCooldownInTooltip() then
				self.tSavedData[KEY_SPELLS][id]["cd"]=""
			else
				if fCooldownTime == 0 or tCharges then
					local nChargeCount = 0
					if tCharges then
						fCooldownTime = tCharges.fRechargeTime
						nChargeCount = tCharges.nRechargeCount
					end
		
					if fCooldownTime == 0 then
						self.tSavedData[KEY_SPELLS][id]["cd"]=Apollo.GetString("Tooltips_NoCooldown")
					elseif fCooldownTime < 60 then
						self.tSavedData[KEY_SPELLS][id]["cd"]=String_GetWeaselString(Apollo.GetString("Tooltips_RechargePerSeconds"), nChargeCount, strRound(fCooldownTime, 0))
					else
						self.tSavedData[KEY_SPELLS][id]["cd"]=String_GetWeaselString(Apollo.GetString("Tooltips_RechargePerMin"), nChargeCount, strRound(fCooldownTime / 60, 1))
					end
				elseif fCooldownTime < 60 then
					self.tSavedData[KEY_SPELLS][id]["cd"]=String_GetWeaselString(Apollo.GetString("Tooltips_SecondsCooldown"), strRound(fCooldownTime, 0))
				else
					self.tSavedData[KEY_SPELLS][id]["cd"]=String_GetWeaselString(Apollo.GetString("Tooltips_MinCooldown"), strRound(fCooldownTime / 60, 1))
				end
			end
		
			-- Mobility
		
			if (eCastMethod == Spell.CodeEnumCastMethod.Normal or eCastMethod == Spell.CodeEnumCastMethod.RapidTap or eCastMethod == Spell.CodeEnumCastMethod.Multiphase) and fCastTime == 0 then
				self.tSavedData[KEY_SPELLS][id]["mob"]=""
			else
				if splSource:IsMovingInterrupted() then
					self.tSavedData[KEY_SPELLS][id]["mob"]=Apollo.GetString("Tooltips_Stationary")
				else
					self.tSavedData[KEY_SPELLS][id]["mob"]=Apollo.GetString("Tooltips_Mobile")
				end
			end

			-- Charges
		
			if tCharges and tCharges.nChargesMax > 1 then
				tAbilityChargeData =
				{
					["name"] = Apollo.GetString("Tooltips_AbilityCharges"),
					["count"] = tCharges.nChargesMax,
				}
			    self.tSavedData[KEY_SPELLS][id]["chgs"]=String_GetWeaselString(Apollo.GetString("Tooltips_Tokens"), tAbilityChargeData)
			else
				self.tSavedData[KEY_SPELLS][id]["chgs"]=""
		    end
		
			self.tSavedData[KEY_SPELLS][id]["tier"]=splSource:GetTier()			
			self.tSavedData[KEY_SPELLS][id]["icon"]=splSource:GetIcon()			
			self.tSavedData[KEY_SPELLS][id]["cls"]=splSource:GetClass()			
			--self.tSavedData[KEY_SPELLS][id]["cp"]=splSource:GetClassPower() removed in reloaded
			--self.tSavedData[KEY_SPELLS][id]["cm"]=splSource:GetCombatMode() removed in reloaded
			self.tSavedData[KEY_SPELLS][id]["sch"]=splSource:GetSchool()		
			self.tSavedData[KEY_SPELLS][id]["ta"]=splSource:GetTargetAngle()		
			self.tSavedData[KEY_SPELLS][id]["rqlvl"]=splSource:GetRequiredLevel()		
			self.tSavedData[KEY_SPELLS][id]["tb"]=nlFix(splSource:GetLasBonusEachTierDesc())
		end
		return id
	end
end

function Jabbithole:AddQuestCategory(cat)
	if cat then
		if self.tSavedData[KEY_QUEST_CATEGORY] == nil then
			self.tSavedData[KEY_QUEST_CATEGORY]={}
		end
		local id=cat:GetId()
		if self.tSavedData[KEY_QUEST_CATEGORY][id] == nil then
			self.tSavedData[KEY_QUEST_CATEGORY][id]=cat:GetTitle()
		end
		return id
	end
	return -1
end

function Jabbithole:AddQuestEpisode(epi)
	if epi then
		if self.tSavedData[KEY_QUEST_EPISODE] == nil then
			self.tSavedData[KEY_QUEST_EPISODE]={}
		end
		local id=epi:GetId()
		if self.tSavedData[KEY_QUEST_EPISODE][id] == nil then
			self.tSavedData[KEY_QUEST_EPISODE][id]=epi:GetTitle()
		end
		return id
	end
	return -1
end

function Jabbithole:AddQuest(quest,srcUnit,srcComm,bSaveSide,bIsRoot)
	if quest then
		if self.tSavedData[KEY_QUESTS] == nil then
			self.tSavedData[KEY_QUESTS]={}
		end
		local id=quest:GetId()
		if self.tSavedData[KEY_QUESTS][id] == nil then
			self.tSavedData[KEY_QUESTS][id]={}
			self.tSavedData[KEY_QUESTS][id]["name"]=quest:GetTitle()
			self.tSavedData[KEY_QUESTS][id]["summary"]=quest:GetSummary()
			self.tSavedData[KEY_QUESTS][id]["xp"]=quest:CalcRewardXP()
			self.tSavedData[KEY_QUESTS][id]["minlevel"]=quest:GetMinLevel()
			self.tSavedData[KEY_QUESTS][id]["level"]=quest:GetConLevel()
			self.tSavedData[KEY_QUESTS][id]["share"]=quest:CanShare()
			self.tSavedData[KEY_QUESTS][id]["abandon"]=quest:CanAbandon()
			self.tSavedData[KEY_QUESTS][id]["difficulty"]=quest:GetColoredDifficulty()
			self.tSavedData[KEY_QUESTS][id]["breadcr"]=quest:IsBreadcrumb()
			self.tSavedData[KEY_QUESTS][id]["zid"]=quest:GetZoneId()
			self.tSavedData[KEY_QUESTS][id]["type"]=quest:GetType()
			self.tSavedData[KEY_QUESTS][id]["stype"]=quest:GetSubType()
			self.tSavedData[KEY_QUESTS][id]["rppr"]=quest:GetRepeatPeriod()
			self.tSavedData[KEY_QUESTS][id]["pthqt"]=quest:GetPathQuestType()
			self.tSavedData[KEY_QUESTS][id]["isatair"]=quest:IsAbleToAdvanceInRaid()
			self.tSavedData[KEY_QUESTS][id]["iscontr"]=quest:IsContract()
			self.tSavedData[KEY_QUESTS][id]["isimb"]=quest:IsImbuementQuest()
			self.tSavedData[KEY_QUESTS][id]["isot"]=quest:IsObjectiveTimed()
			self.tSavedData[KEY_QUESTS][id]["isoaipvp"]=quest:IsOnlyAdvanceInPvp()
			self.tSavedData[KEY_QUESTS][id]["isoptfe"]=quest:IsOptionalForEpisode()
			self.tSavedData[KEY_QUESTS][id]["ispath"]=quest:IsPathQuest()
			self.tSavedData[KEY_QUESTS][id]["istimed"]=quest:IsQuestTimed()
			
--			self.tSavedData[KEY_QUESTS][id]["zone"]=self.tZone.id
			self.tSavedData[KEY_QUESTS][id]["npc"]=nil
			self.tSavedData[KEY_QUESTS][id]["posx"]=nil
			self.tSavedData[KEY_QUESTS][id]["posy"]=nil
			self.tSavedData[KEY_QUESTS][id]["callzone"]={}
			self.tSavedData[KEY_QUESTS][id]["start"]={}
			self.tSavedData[KEY_QUESTS][id]["finish"]={}
			
			if quest:GetCategory() then
				self.tSavedData[KEY_QUESTS][id]["category"]=self:AddQuestCategory(quest:GetCategory())
			end

			if quest:GetEpisode() then
				self.tSavedData[KEY_QUESTS][id]["episode"]=self:AddQuestEpisode(quest:GetEpisode())
			end
	
			self.tSavedData[KEY_QUESTS][id]["objectives"]={}
			if quest:GetObjectiveCount()>0 then
				for idx=0, quest:GetObjectiveCount()-1 do
					self.tSavedData[KEY_QUESTS][id]["objectives"][idx]=quest:GetObjectiveDescription(idx)
				end
			end

			if bSaveSide then
				if self.tPlayerFaction == Unit.CodeEnumFaction.DominionPlayer then
					self.tSavedData[KEY_QUESTS][id]["side_d"]=true
				end
				if self.tPlayerFaction == Unit.CodeEnumFaction.ExilesPlayer then
					self.tSavedData[KEY_QUESTS][id]["side_e"]=true
				end
			end
							
			local fixItems = {}
			local optItems = {}
			local fixCash = {}
			local optCash = {}
			local fixRep = {}
			local optRep = {}
			local fixTradeskill = {}
			local optTradeskill = {}
			
			local rewards = quest:GetRewardData()
			
			--if tGivenRewards and #tGivenRewards > 0 then
				for key, tCurrReward in ipairs(rewards.arFixedRewards or {}) do
					if tCurrReward and tCurrReward.eType == Quest.Quest2RewardType_Item then
						fixItems [tCurrReward.idReward] = tCurrReward.nAmount.."/"..self:AddItem(tCurrReward.itemReward, id, nil, bSaveSide, bIsRoot)
					end
					if tCurrReward and tCurrReward.eType == Quest.Quest2RewardType_Money then
						if tCurrReward.eCurrencyType or tCurrReward.idObject then
							if tCurrReward.eCurrencyType then
								fixCash[tCurrReward.idReward] = tCurrReward.nAmount.."/"..tCurrReward.eCurrencyType
							else
								fixCash[tCurrReward.idReward] = tCurrReward.nAmount.."/"..tCurrReward.idObject 
							end
						end
					end
					if tCurrReward and tCurrReward.eType == Quest.Quest2RewardType_TradeSkillXp then
						fixTradeskill [tCurrReward.idReward] = tCurrReward.nXP.."/"..tCurrReward.idObject.."/"..tCurrReward.strTradeskill
					end
					if tCurrReward and tCurrReward.eType == Quest.Quest2RewardType_Reputation then
						fixRep [tCurrReward.idObject] = tCurrReward.nAmount.."/"..tCurrReward.strFactionName
					end
				end
			--end
	
			--if tChoiceRewards and #tChoiceRewards > 0 then
				for key, tCurrReward in ipairs(rewards.arRewardChoices or {}) do
					if tCurrReward and tCurrReward.eType == Quest.Quest2RewardType_Item then
						optItems [tCurrReward.idReward] = tCurrReward.nAmount.."/"..self:AddItem(tCurrReward.itemReward, id, nil, bSaveSide, bIsRoot)
					end
					if tCurrReward and tCurrReward.eType == Quest.Quest2RewardType_Money then
						if tCurrReward.eCurrencyType or tCurrReward.idObject then
							if tCurrReward.eCurrencyType then
								optCash[tCurrReward.idReward] = tCurrReward.nAmount.."/"..tCurrReward.eCurrencyType
							else
								optCash[tCurrReward.idReward] = tCurrReward.nAmount.."/"..tCurrReward.idObject 
							end
						end
					end
					if tCurrReward and tCurrReward.eType == Quest.Quest2RewardType_TradeSkillXp then
						optTradeskill [tCurrReward.idReward] = tCurrReward.nXP.."/"..tCurrReward.idObject.."/"..tCurrReward.strTradeskill
					end
					if tCurrReward and tCurrReward.eType == Quest.Quest2RewardType_Reputation then
						optRep [tCurrReward.idObject] = tCurrReward.nAmount.."/"..tCurrReward.strFactionName
					end
				end
			--end
				
			self.tSavedData[KEY_QUESTS][id]["fixItems"]=fixItems
			self.tSavedData[KEY_QUESTS][id]["optItems"]=optItems
			self.tSavedData[KEY_QUESTS][id]["fixCash"]=fixCash
			self.tSavedData[KEY_QUESTS][id]["optCash"]=optCash
			self.tSavedData[KEY_QUESTS][id]["fixTradeskill"]=fixTradeskill
			self.tSavedData[KEY_QUESTS][id]["optTradeskill"]=optTradeskill
			self.tSavedData[KEY_QUESTS][id]["fixRep"]=fixRep
			self.tSavedData[KEY_QUESTS][id]["optRep"]=optRep
		else
			self.tSavedData[KEY_QUESTS][id]["breadcr"]=quest:IsBreadcrumb()
			self.tSavedData[KEY_QUESTS][id]["zid"]=quest:GetZoneId()
			self.tSavedData[KEY_QUESTS][id]["type"]=quest:GetType()
			self.tSavedData[KEY_QUESTS][id]["stype"]=quest:GetSubType()
			self.tSavedData[KEY_QUESTS][id]["rppr"]=quest:GetRepeatPeriod()
			self.tSavedData[KEY_QUESTS][id]["pthqt"]=quest:GetPathQuestType()

			if self.tSavedData[KEY_QUESTS][id]["fixItems"] == nil then
				self.tSavedData[KEY_QUESTS][id]["fixItems"]={}
			end
			if self.tSavedData[KEY_QUESTS][id]["optItems"] == nil then
				self.tSavedData[KEY_QUESTS][id]["optItems"]={}
			end
			if self.tSavedData[KEY_QUESTS][id]["fixCash"] == nil then
				self.tSavedData[KEY_QUESTS][id]["fixCash"]={}
			end
			if self.tSavedData[KEY_QUESTS][id]["optCash"] == nil then
				self.tSavedData[KEY_QUESTS][id]["optCash"]={}
			end
			if self.tSavedData[KEY_QUESTS][id]["fixTradeskill"] == nil then
				self.tSavedData[KEY_QUESTS][id]["fixTradeskill"]={}
			end
			if self.tSavedData[KEY_QUESTS][id]["optTradeskill"] == nil then
				self.tSavedData[KEY_QUESTS][id]["optTradeskill"]={}
			end
			if self.tSavedData[KEY_QUESTS][id]["fixRep"] == nil then
				self.tSavedData[KEY_QUESTS][id]["fixRep"]={}
			end
			if self.tSavedData[KEY_QUESTS][id]["optRep"] == nil then
				self.tSavedData[KEY_QUESTS][id]["optRep"]={}
			end
		
--			Print("existing quest")
			if self.tPlayerFaction == Unit.CodeEnumFaction.DominionPlayer then
				self.tSavedData[KEY_QUESTS][id]["side_d"]=true
			end
			if self.tPlayerFaction == Unit.CodeEnumFaction.ExilesPlayer then
				self.tSavedData[KEY_QUESTS][id]["side_e"]=true
			end
			
--			Print(""..#quest:GetRewardData())
			
			-- update item rewards for filtered reward set (per class stuff)
			local rewards = quest:GetRewardData()

--			if tGivenRewards and #tGivenRewards > 0 then
				for key, tCurrReward in ipairs(rewards.arFixedRewards or {}) do
					if tCurrReward and tCurrReward.eType == Quest.Quest2RewardType_Item then
						self.tSavedData[KEY_QUESTS][id]["fixItems"][tCurrReward.idReward] = tCurrReward.nAmount.."/"..self:AddItem(tCurrReward.itemReward, id, nil, bSaveSide, bIsRoot)
					end
				end
--			end	
--			if tChoiceRewards and #tChoiceRewards > 0 then
				for key, tCurrReward in ipairs(rewards.arRewardChoices or {}) do
					if tCurrReward and tCurrReward.eType == Quest.Quest2RewardType_Item then
						self.tSavedData[KEY_QUESTS][id]["optItems"][tCurrReward.idReward] = tCurrReward.nAmount.."/"..self:AddItem(tCurrReward.itemReward, id, nil, bSaveSide, bIsRoot)
					end
				end
--			end
		end
		--merges
		if srcUnit and srcUnit:IsValid() then
			local starter=self:MakeCreatureId(srcUnit)

			if self.tSavedData[KEY_QUESTS][id]["start"]==nil then
				self.tSavedData[KEY_QUESTS][id]["start"]={}
			end
			if self.tSavedData[KEY_QUESTS][id]["start"][starter]==nil then
				self.tSavedData[KEY_QUESTS][id]["start"][starter]={}
			end
			
			self.tSavedData[KEY_QUESTS][id]["start"][starter][srcUnit:GetPosition().x.."/"..srcUnit:GetPosition().y.."/"..srcUnit:GetPosition().z]=1
			self:AddCreature(srcUnit)
		end
		if srcComm then
			if self.tSavedData[KEY_QUESTS][id]["callzone"]==nil then
				self.tSavedData[KEY_QUESTS][id]["callzone"]={}
			end
			if self.tSavedData[KEY_QUESTS][id]["callzone"][self.tZone.id]==nil then
				self.tSavedData[KEY_QUESTS][id]["callzone"][self.tZone.id]={}
			end
			if self.tPlayer == nil then
				self.tPlayer=GameLib.GetPlayerUnit()
			end
			self.tSavedData[KEY_QUESTS][id]["callzone"][self.tZone.id][self.tPlayer:GetPosition().x.."/"..self.tPlayer:GetPosition().y.."/"..self.tPlayer:GetPosition().z]=1
		end
	end
end

function Jabbithole:AddZone(zone)
	if zone then
		if self.tSavedData[KEY_ZONES] == nil then
			self.tSavedData[KEY_ZONES]={}
		end
		local id=zone.id
		if self.tSavedData[KEY_ZONES][id] == nil then
			self.tSavedData[KEY_ZONES][id]={}
			self.tSavedData[KEY_ZONES][id]["name"]=zone.strName
			self.tSavedData[KEY_ZONES][id]["map"]=zone.strFolder
			self.tSavedData[KEY_ZONES][id]["continent"]=zone.continentId
			self.tSavedData[KEY_ZONES][id]["bounds"]=zone.fNorth.."/"..zone.fEast.."/"..zone.fSouth.."/"..zone.fWest
		end
	end
end

function Jabbithole:AddCreatureSpellRelation(unit,spell)
	if unit and spell and not unit:IsACharacter() then
		if self.tSavedData[KEY_CREATURE_SPELL] == nil then
			self.tSavedData[KEY_CREATURE_SPELL]={}
		end
		local cid=self:MakeCreatureId(unit)
		if self.tSavedData[KEY_CREATURES][cid] then
			local sid=spell:GetId()
			self.tSavedData[KEY_CREATURE_SPELL][sid.."/"..cid]=1
		end
	end
end

function Jabbithole:MakeCreatureId(unit)
	local n=unit:GetName()
	if n==nil or n=="" then
		n="Name not specified"
	end
	local ret=self.tZone.id.."/"..n
	bVeteranMode = (GameLib.GetWorldDifficulty()==GroupLib.Difficulty.Veteran)
	if bVeteranMode then
		if unit:GetMaxHealth() then
			ret = ret .. "/v"
		else
			local ufac=unit:GetFacing()
			if ufac then
				if ufac.x==-0 and ufac.y==0 and ufac.z==-1 then
					ret = ret .. "/v"
				end
			end
		end
	end
	return ret
end

function Jabbithole:AddCreature(unit,checkForInvLooter)
	if unit and (not unit:IsACharacter()) and unit:IsValid() then
		if unit:GetUnitOwner() then
			if unit:GetUnitOwner():IsACharacter() then
				return
			end
		end
		if unit:GetType()=="Scanner" or unit:GetType()=="PinataLoot" or unit:GetType()=="Ghost" or unit:GetType()=="Pickup" or unit:GetType()=="Pet" or unit:GetType()=="Mount" then
			return
		end
	
		if self.tSavedData[KEY_CREATURES] == nil then
			self.tSavedData[KEY_CREATURES]={}
		end
		
		local id=self:MakeCreatureId(unit)
		
		local upos=unit:GetPosition()
		if self.tSavedData[KEY_CREATURES][id] == nil then
			self.tSavedData[KEY_CREATURES][id]={}
			self.tSavedData[KEY_CREATURES][id]["name"]=unit:GetName()
			self.tSavedData[KEY_CREATURES][id]["zone"]={}
			
			if self.tPlayerFaction == Unit.CodeEnumFaction.DominionPlayer then
				self.tSavedData[KEY_CREATURES][id]["dispd"]=unit:GetDispositionTo(self.tPlayer)
			end
			if self.tPlayerFaction == Unit.CodeEnumFaction.ExilesPlayer then
				self.tSavedData[KEY_CREATURES][id]["dispe"]=unit:GetDispositionTo(self.tPlayer)
			end
			self.tSavedData[KEY_CREATURES][id]["aff"]=unit:GetAffiliationName()
			self.tSavedData[KEY_CREATURES][id]["clsid"]=unit:GetClassId()
			self.tSavedData[KEY_CREATURES][id]["diff"]=unit:GetDifficulty()
			self.tSavedData[KEY_CREATURES][id]["eli"]=unit:GetEliteness()
			self.tSavedData[KEY_CREATURES][id]["fact"]=unit:GetFaction()
			self.tSavedData[KEY_CREATURES][id]["sh"]=unit:GetShieldCapacityMax()
			self.tSavedData[KEY_CREATURES][id]["nplc"]=unit:GetNameplateColor()
			self.tSavedData[KEY_CREATURES][id]["race"]=unit:GetRaceId()
			self.tSavedData[KEY_CREATURES][id]["type"]=unit:GetType()
			self.tSavedData[KEY_CREATURES][id]["rank"]=unit:GetRank()
			if unit:GetArchetype() then
				self.tSavedData[KEY_CREATURES][id]["at_id"]=unit:GetArchetype()["idArchetype"]
				self.tSavedData[KEY_CREATURES][id]["at_ic"]=unit:GetArchetype()["strIcon"]
			end
			self.tSavedData[KEY_CREATURES][id]["sex"]=unit:GetGender()
			
--initialized on demand, saves space
--[[			self.tSavedData[KEY_CREATURES][id]["q"]={}
			self.tSavedData[KEY_CREATURES][id]["pe"]={}
			self.tSavedData[KEY_CREATURES][id]["chg"]={}]]
			self.tSavedData[KEY_CREATURES][id]["lvlmin"]=unit:GetLevel()
			self.tSavedData[KEY_CREATURES][id]["lvlmax"]=unit:GetLevel()
			self.tSavedData[KEY_CREATURES][id]["hpmin"]=unit:GetMaxHealth()
			self.tSavedData[KEY_CREATURES][id]["hpmax"]=unit:GetMaxHealth()
		end
		
		self.tSavedData[KEY_CREATURES][id]["wid"]=self.worldId
		self.tSavedData[KEY_CREATURES][id]["wzid"]=self.worldZoneId
		self.tSavedData[KEY_CREATURES][id]["pwrass"]=unit:GetAssaultPower()
		self.tSavedData[KEY_CREATURES][id]["pwrsup"]=unit:GetSupportPower()
		self.tSavedData[KEY_CREATURES][id]["crtcha"]=unit:GetCritChance()
		self.tSavedData[KEY_CREATURES][id]["crtsev"]=unit:GetCritSeverity()
		self.tSavedData[KEY_CREATURES][id]["dflcha"]=unit:GetDeflectChance()
		self.tSavedData[KEY_CREATURES][id]["dflcrtcha"]=unit:GetDeflectCritChance()
		self.tSavedData[KEY_CREATURES][id]["inta"]=unit:GetInterruptArmorMax()
		self.tSavedData[KEY_CREATURES][id]["rare"]=unit:IsRare()


		-- merge stuff
		
--[[		local tMarkerInfoList = self:GetOrderedMarkerInfos(tMarkers)
		for nIdx, tMarkerInfo in ipairs(tMarkerInfoList) do
			local tInfo = self:GetDefaultUnitInfo()
			if tMarkerInfo.strIcon  then
			tInfo.strIcon = tMarkerInfo.strIcon
		end

		self.tSavedData[KEY_CREATURES][id]["mmm"]=unit:GetMiniMapMarker()]]
		
		if checkForInvLooter then
			local fac=unit:GetFacing()
			if fac then
				if fac.x==-0 and fac.y==0 and fac.z==-1 then
					self.tSavedData[KEY_CREATURES][id]["looter"]=1
				end
			end
		end
		
		--lvl and hp range
		if unit:GetLevel() then
			if self.tSavedData[KEY_CREATURES][id]["lvlmin"] then
				if tonumber(unit:GetLevel())<tonumber(self.tSavedData[KEY_CREATURES][id]["lvlmin"]) then
					self.tSavedData[KEY_CREATURES][id]["lvlmin"]=unit:GetLevel()
				end
			else
				self.tSavedData[KEY_CREATURES][id]["lvlmin"]=unit:GetLevel()
			end
			if self.tSavedData[KEY_CREATURES][id]["lvlmax"] then
				if tonumber(unit:GetLevel())>tonumber(self.tSavedData[KEY_CREATURES][id]["lvlmax"]) then
					self.tSavedData[KEY_CREATURES][id]["lvlmax"]=unit:GetLevel()
				end
			else
				self.tSavedData[KEY_CREATURES][id]["lvlmax"]=unit:GetLevel()
			end
		end
		if unit:GetMaxHealth() then
			if self.tSavedData[KEY_CREATURES][id]["hpmin"] then
				if tonumber(unit:GetMaxHealth())<tonumber(self.tSavedData[KEY_CREATURES][id]["hpmin"]) then
					self.tSavedData[KEY_CREATURES][id]["hpmin"]=unit:GetMaxHealth()
				end
			else
				self.tSavedData[KEY_CREATURES][id]["hpmin"]=unit:GetMaxHealth()
			end
			if self.tSavedData[KEY_CREATURES][id]["hpmax"] then
				if tonumber(unit:GetMaxHealth())>tonumber(self.tSavedData[KEY_CREATURES][id]["hpmax"]) then
					self.tSavedData[KEY_CREATURES][id]["hpmax"]=unit:GetMaxHealth()
				end
			else
				self.tSavedData[KEY_CREATURES][id]["hpmax"]=unit:GetMaxHealth()
			end
		end
		
		--location
		if self.tSavedData[KEY_CREATURES][id]["zone"][upos.x.."/"..upos.y.."/"..upos.z] == nil then
			if creatureLocationCached[unit:GetId()] == nil then
				self.tSavedData[KEY_CREATURES][id]["zone"][upos.x.."/"..upos.y.."/"..upos.z]=1
				creatureLocationCached[unit:GetId()]=1
			end
		end
		--keysets
--[[		local keyset=self.tSavedData[KEY_CREATURES][id]["as"]
		if keyset==nil then
			keyset=""
		end]]
		local as = unit:GetActivationState()
		lastDatacube = nil
		if as then
			if self.tSavedData[KEY_CREATURES][id]["as2"] == nil then
				self.tSavedData[KEY_CREATURES][id]["as2"]={}
			end
			for k,v in pairs(as) do
				self.tSavedData[KEY_CREATURES][id]["as2"][k]=1
--[[				if keyset ~= "" then
					keyset=keyset..","
				end
				keyset=keyset..k]]
				if k=="Datacube" then
					lastDatacube = id
				end
			end
		end
--		self.tSavedData[KEY_CREATURES][id]["as"]=keyset
		--rewardinfo
		local ri = unit:GetRewardInfo()
		if self.tSavedData[KEY_CREATURES][id]["q"]==nil then
			self.tSavedData[KEY_CREATURES][id]["q"]={}
		end
		if self.tSavedData[KEY_CREATURES][id]["qc"]==nil then
			self.tSavedData[KEY_CREATURES][id]["qc"]={}
		end
		if self.tSavedData[KEY_CREATURES][id]["pe"]==nil then
			self.tSavedData[KEY_CREATURES][id]["pe"]={}
		end
		if self.tSavedData[KEY_CREATURES][id]["chg"]==nil then
			self.tSavedData[KEY_CREATURES][id]["chg"]={}
		end
		if ri then
			for k,v in pairs(ri) do
				if v.eType == Unit.CodeEnumRewardInfoType.Quest then
					if self.tSavedData[KEY_CREATURES][id]["q"]==nil then
						self.tSavedData[KEY_CREATURES][id]["q"]={}
					end
					self.tSavedData[KEY_CREATURES][id]["q"][v.idQuest]=1
				end

				if v.eType == Unit.CodeEnumRewardInfoType.Contract then
					if self.tSavedData[KEY_CREATURES][id]["qc"]==nil then
						self.tSavedData[KEY_CREATURES][id]["qc"]={}
					end
					self.tSavedData[KEY_CREATURES][id]["qc"][v.idQuest]=1
				end


				if v.eType == Unit.CodeEnumRewardInfoType.PublicEvent then
					if self.tSavedData[KEY_CREATURES][id]["pe"]==nil then
						self.tSavedData[KEY_CREATURES][id]["pe"]={}
					end
					local peo=v.peoObjective
					local pe=v.peoObjective:GetEvent()
					
					local peid=self:AddPublicEvent(pe)
					if peid ~= -1 then
--					local peid=self.tZone.id.."/"..pe:GetName()
						self.tSavedData[KEY_CREATURES][id]["pe"][peid]=1
						if self.tSavedData[KEY_PUBLIC_EVENT][peid]["realzone"]==nil then
							self.tSavedData[KEY_PUBLIC_EVENT][peid]["realzone"]=self.tZone.id
						else
							if self.tSavedData[KEY_PUBLIC_EVENT][peid]["realzone"] ~= self.tZone.id then
								self.tSavedData[KEY_PUBLIC_EVENT][peid]["realzone"] = -1
							end
						end
					end
				end
				
				if v.eType == Unit.CodeEnumRewardInfoType.Challenge then
					if self.tSavedData[KEY_CREATURES][id]["chg"]==nil then
						self.tSavedData[KEY_CREATURES][id]["chg"]={}
					end
					self.tSavedData[KEY_CREATURES][id]["chg"][v.idChallenge]=1
				end
				
				if v.eType == Unit.CodeEnumRewardInfoType.Scientist then
					self.tSavedData[KEY_CREATURES][id]["p_sci"]=true
					if v.pmMission then
						if self.tSavedData[KEY_CREATURES][id]["pathm"]==nil then
							self.tSavedData[KEY_CREATURES][id]["pathm"]={}
						end
						self.tSavedData[KEY_CREATURES][id]["pathm"][v.pmMission:GetId()]=1
--[[						if self.tSavedData[KEY_PATH_MISSION][v.pmMission:GetId()] ~= nil then
							if self.tSavedData[KEY_PATH_MISSION][v.pmMission:GetId()]["rz"] == nil then
								self.tSavedData[KEY_PATH_MISSION][v.pmMission:GetId()]["rz"]=self.tZone.id
							else
								if self.tSavedData[KEY_PATH_MISSION][v.pmMission:GetId()]["rz"] ~= self.tZone.id then
									self.tSavedData[KEY_PATH_MISSION][v.pmMission:GetId()]["rz"] = -1
								end
							end
						end]]
					end
					if v.splReward then
						if self.tSavedData[KEY_CREATURES][id]["spell_sci"]==nil then
							self.tSavedData[KEY_CREATURES][id]["spell_sci"]={}
						end
						self:AddSpell(v.splReward)
						self.tSavedData[KEY_CREATURES][id]["spell_sci"][v.splReward:GetId()]=1
					end
				end
				if v.eType == Unit.CodeEnumRewardInfoType.Soldier then
					self.tSavedData[KEY_CREATURES][id]["p_sol"]=true
					if v.pmMission then
						if self.tSavedData[KEY_CREATURES][id]["pathm"]==nil then
							self.tSavedData[KEY_CREATURES][id]["pathm"]={}
						end
						self.tSavedData[KEY_CREATURES][id]["pathm"][v.pmMission:GetId()]=1
--[[						if self.tSavedData[KEY_PATH_MISSION][v.pmMission:GetId()] ~= nil then
							if self.tSavedData[KEY_PATH_MISSION][v.pmMission:GetId()]["rz"] == nil then
								self.tSavedData[KEY_PATH_MISSION][v.pmMission:GetId()]["rz"]=self.tZone.id
							else
								if self.tSavedData[KEY_PATH_MISSION][v.pmMission:GetId()]["rz"] ~= self.tZone.id then
									self.tSavedData[KEY_PATH_MISSION][v.pmMission:GetId()]["rz"] = -1
								end
							end
						end]]
					end
				end
				if v.eType == Unit.CodeEnumRewardInfoType.Settler then
					self.tSavedData[KEY_CREATURES][id]["p_set"]=true
					if v.pmMission then
						if self.tSavedData[KEY_CREATURES][id]["pathm"]==nil then
							self.tSavedData[KEY_CREATURES][id]["pathm"]={}
						end
						self.tSavedData[KEY_CREATURES][id]["pathm"][v.pmMission:GetId()]=1
--[[						if self.tSavedData[KEY_PATH_MISSION][v.pmMission:GetId()] ~= nil then
							if self.tSavedData[KEY_PATH_MISSION][v.pmMission:GetId()]["rz"] == nil then
								self.tSavedData[KEY_PATH_MISSION][v.pmMission:GetId()]["rz"]=self.tZone.id
							else
								if self.tSavedData[KEY_PATH_MISSION][v.pmMission:GetId()]["rz"] ~= self.tZone.id then
									self.tSavedData[KEY_PATH_MISSION][v.pmMission:GetId()]["rz"] = -1
								end
							end
						end]]
					end
				end
				if v.eType == Unit.CodeEnumRewardInfoType.Explorer then
					self.tSavedData[KEY_CREATURES][id]["p_exp"]=true
					if v.pmMission then
						if self.tSavedData[KEY_CREATURES][id]["pathm"]==nil then
							self.tSavedData[KEY_CREATURES][id]["pathm"]={}
						end
						self.tSavedData[KEY_CREATURES][id]["pathm"][v.pmMission:GetId()]=1
--[[						if self.tSavedData[KEY_PATH_MISSION][v.pmMission:GetId()] ~= nil then
							if self.tSavedData[KEY_PATH_MISSION][v.pmMission:GetId()]["rz"] == nil then
								self.tSavedData[KEY_PATH_MISSION][v.pmMission:GetId()]["rz"]=self.tZone.id
							else
								if self.tSavedData[KEY_PATH_MISSION][v.pmMission:GetId()]["rz"] ~= self.tZone.id then
									self.tSavedData[KEY_PATH_MISSION][v.pmMission:GetId()]["rz"] = -1
								end
							end
						end]]
					end
				end
			end
		end
	end
end

function Jabbithole:AddTitle(title, category, full)
	if title and category then
		if self.tSavedData[KEY_TITLES] == nil then
			self.tSavedData[KEY_TITLES]={}
		end

		if self.tPlayer == nil then
			self.tPlayer = GameLib.GetPlayerUnit()
		end
		
		local key=title.."//"..category
		if self.tSavedData[KEY_TITLES][key] == nil then
			if self.tPlayer:GetName() ~= "" then
				--TODO "" names
				self.tSavedData[KEY_TITLES][key]={}
			end
		end

		if self.tSavedData[KEY_TITLES][key]~=nil then
			local n = self.tPlayer:GetName()
			if n and n ~= "" then
				local len=string.len(n)
				local pos=string.find(full,n.." ")
				if pos==1 then
				  self.tSavedData[KEY_TITLES][key]["title"]="<name> "..string.sub(full, len+2)
				else
				  pos=string.find(full,n..", ")
				  if pos==1 then
				    self.tSavedData[KEY_TITLES][key]["title"]="<name>, "..string.sub(full, len+3)
				  else
					if pos==string.len(full)-string.len(n)+1 then
						self.tSavedData[KEY_TITLES][key]["title"]=string.sub(full, 1, string.len(full)-len-1).." <name>"
					else
						local t,_ = full:gsub(n,"<name>")
						self.tSavedData[KEY_TITLES][key]["title"]=t
					end
				  end
				end				
			end
			if self.tPlayerFaction == Unit.CodeEnumFaction.DominionPlayer then
				self.tSavedData[KEY_TITLES][key]["side_d"]=true
			end
			if self.tPlayerFaction == Unit.CodeEnumFaction.ExilesPlayer then
				self.tSavedData[KEY_TITLES][key]["side_e"]=true
			end
		end
		
	end
end

function Jabbithole:AddDatacube(id, isvolume, datacube)
	if datacube then
		if self.tSavedData[KEY_DATACUBE] == nil then
			self.tSavedData[KEY_DATACUBE]={}
		end
		
		if self.tSavedData[KEY_DATACUBE][id] == nil then
			self.tSavedData[KEY_DATACUBE][id]={}
			self.tSavedData[KEY_DATACUBE][id]["npc"]={}
		end
		self.tSavedData[KEY_DATACUBE][id]["dctype"]=datacube.eDatacubeType
		self.tSavedData[KEY_DATACUBE][id]["dcid"]=datacube.nDatacubeId
		self.tSavedData[KEY_DATACUBE][id]["volid"]=datacube.nVolumeId
		self.tSavedData[KEY_DATACUBE][id]["wzid"]=datacube.nWorldZoneId
		self.tSavedData[KEY_DATACUBE][id]["title"]=datacube.strTitle
		self.tSavedData[KEY_DATACUBE][id]["text"]=datacube.strText
		self.tSavedData[KEY_DATACUBE][id]["isvol"]="-"
		if isvolume then
			self.tSavedData[KEY_DATACUBE][id]["isvol"]="+"
		end
		if lastDatacube then
			self.tSavedData[KEY_DATACUBE][id]["npc"][lastDatacube]=1
			lastDatacube=nil
		end
	end
end

function Jabbithole:AddChallenge(chg)
	if chg then
		if self.tSavedData[KEY_CHALLENGE] == nil then
			self.tSavedData[KEY_CHALLENGE]={}
		end
		
		local id=chg:GetId()
		if self.tSavedData[KEY_CHALLENGE][id] == nil then
			self.tSavedData[KEY_CHALLENGE][id]={}
		end
		
		if self.tSavedData[KEY_CHALLENGE][id]["loc"]==nil then
			self.tSavedData[KEY_CHALLENGE][id]["loc"]={}
		end
		if self.tSavedData[KEY_CHALLENGE][id]["sloc"]==nil then
			self.tSavedData[KEY_CHALLENGE][id]["sloc"]={}
		end
-- changed in reloaded
--		if self.tSavedData[KEY_CHALLENGE][id]["rew"]==nil then
--			self.tSavedData[KEY_CHALLENGE][id]["rew"]={}
--		end

		local zone=chg:GetZoneInfo()
		if zone then
			self.tSavedData[KEY_CHALLENGE][id]["zone"]=zone.idZone
		else
			self.tSavedData[KEY_CHALLENGE][id]["zone"]=-1
		end
		self.tSavedData[KEY_CHALLENGE][id]["name"]=chg:GetName()
		self.tSavedData[KEY_CHALLENGE][id]["type"]=chg:GetType()
		self.tSavedData[KEY_CHALLENGE][id]["desc"]=chg:GetDescription()
		self.tSavedData[KEY_CHALLENGE][id]["timed"]=chg:IsTimeTiered()
		local goals=""
		for idx,goal in pairs(chg:GetAllTierCounts() or {}) do
			if idx>1 then
				goals=goals.."/"
			end
			goals=goals..goal.nGoalCount
		end
		self.tSavedData[KEY_CHALLENGE][id]["goals"]=goals

		--merge
		if self.tPlayerFaction == Unit.CodeEnumFaction.DominionPlayer then
			self.tSavedData[KEY_CHALLENGE][id]["side_d"]=true
		end
		if self.tPlayerFaction == Unit.CodeEnumFaction.ExilesPlayer then
			self.tSavedData[KEY_CHALLENGE][id]["side_e"]=true
		end
		
		local loc = chg:GetMapLocation()
		if loc then
			self.tSavedData[KEY_CHALLENGE][id]["loc"][loc.x.."/"..loc.y.."/"..loc.z]=1
		end
		loc = chg:GetMapStartLocation()
		if loc then
			self.tSavedData[KEY_CHALLENGE][id]["sloc"][loc.x.."/"..loc.y.."/"..loc.z]=1
		end
		local zri=chg:GetZoneRestrictionInfo()
		if zri then
			self.tSavedData[KEY_CHALLENGE][id]["zri"]=zri.idSubZone
		end
		
		self.tSavedData[KEY_CHALLENGE][id]["rew"]=self:AddChallengeRewardTrack(self.tSavedData[KEY_CHALLENGE][id]["zone"], chg:GetRewardTrack())
	end
end

function Jabbithole:AddChallengeRewardTrack(zoneid, rewardTrack)
	if rewardTrack and rewardTrack:GetType() == RewardTrackLib.CodeEnumRewardTrackType.Challenge then

		if self.tSavedData[KEY_CHALLENGE_REWARDS] == nil then
			self.tSavedData[KEY_CHALLENGE_REWARDS]={}
		end
		
		local id=rewardTrack:GetId()
		if self.tSavedData[KEY_CHALLENGE_REWARDS][id] == nil then
			self.tSavedData[KEY_CHALLENGE_REWARDS][id]={}
			self.tSavedData[KEY_CHALLENGE_REWARDS][id]["rew"]={}
		end

		self.tSavedData[KEY_CHALLENGE_REWARDS][id]["trid"]=rewardTrack:GetId()
		self.tSavedData[KEY_CHALLENGE_REWARDS][id]["trass"]=rewardTrack:GetImageAssetPath()
		self.tSavedData[KEY_CHALLENGE_REWARDS][id]["trtyp"]=rewardTrack:GetType()
		
		for idx, rew in pairs(rewardTrack:GetAllRewards() or {}) do
			self.tSavedData[KEY_CHALLENGE_REWARDS][id]["rew"][rew.idReward]={}
			self.tSavedData[KEY_CHALLENGE_REWARDS][id]["rew"][rew.idReward]["casq"]=rew.bIsCasque		
			self.tSavedData[KEY_CHALLENGE_REWARDS][id]["rew"][rew.idReward]["cost"]=rew.nCost
			self.tSavedData[KEY_CHALLENGE_REWARDS][id]["rew"][rew.idReward]["rewidx"]=rew.nRewardIdx
			if self.tPlayerFaction == Unit.CodeEnumFaction.DominionPlayer then
				self.tSavedData[KEY_CHALLENGE_REWARDS][id]["rew"][rew.idReward]["side_d"]=true
			end
			if self.tPlayerFaction == Unit.CodeEnumFaction.ExilesPlayer then
				self.tSavedData[KEY_CHALLENGE_REWARDS][id]["rew"][rew.idReward]["side_e"]=true
			end
			self.tSavedData[KEY_CHALLENGE_REWARDS][id]["rew"][rew.idReward]["items"]={}
			for idx2, choice in pairs(rew.tRewardChoices or {}) do
				if choice.itemReward then
					local iid=self:AddItem(choice.itemReward,-1,nil,true)
					self.tSavedData[KEY_CHALLENGE_REWARDS][id]["rew"][rew.idReward]["items"][iid]={}
					self.tSavedData[KEY_CHALLENGE_REWARDS][id]["rew"][rew.idReward]["items"][iid]["rt"]=choice.eRewardType
					self.tSavedData[KEY_CHALLENGE_REWARDS][id]["rew"][rew.idReward]["items"][iid]["idx"]=choice.nChoiceIdx
					self.tSavedData[KEY_CHALLENGE_REWARDS][id]["rew"][rew.idReward]["items"][iid]["cnt"]=choice.nRewardCount
				end
			end
		end
		return id
	else
		return nil
	end
end

function Jabbithole:AddChallenges()
	for idx,chg in pairs(ChallengesLib.GetActiveChallengeList() or {}) do
		self:AddChallenge(chg)
	end
end

function Jabbithole:OnChallengesTimer()
	if GameLib.GetPlayerUnit() and GameLib.GetPlayerUnit():GetFaction() then
		Apollo.StopTimer("ChallengesScanner")
		self:AddChallenges()
	else
		Apollo.StartTimer("ChallengesScanner")	
	end
end

--[[ obsolete in reloaded
function Jabbithole:AddChallengeReward(chgid, reward)
	if reward~=nil and reward.itemReward~=nil then
		local id=chgid
		if self.tSavedData[KEY_CHALLENGE][id] ~= nil then
			self.tSavedData[KEY_CHALLENGE][id]["rew"][reward.nRewardId]=reward.nAmount.."/"..self:AddItem(reward.itemReward,-1,nil,true).."/"..reward.nChallengeTier
		end
	end
end
]]

function Jabbithole:FindPublicEvent(e)
	local game_id=-1

	for idx,pe in pairs(PublicEventsLib.GetActivePublicEventList() or {}) do
		if pe then
			if pe:GetName()==e:GetName() then
				game_id=idx
			end
		end
	end

	return game_id
end

function Jabbithole:AddPublicEventMission(peid, name, desc)
	if self.tSavedData[KEY_PUBLIC_EVENT][peid]["miss"] == nil then
		self.tSavedData[KEY_PUBLIC_EVENT][peid]["miss"]={}
	end
	self.tSavedData[KEY_PUBLIC_EVENT][peid]["miss"][name]=desc
end

function Jabbithole:AddPublicEvent(e,updateRZ)
	local game_id=-1
	if self.bInitializedProperly then
	if e then
		if self.tSavedData[KEY_PUBLIC_EVENT] == nil then
			self.tSavedData[KEY_PUBLIC_EVENT]={}
		end
		
		game_id=self:FindPublicEvent(e)
		
		if game_id ~= -1 then
			local id=game_id
			if self.tSavedData[KEY_PUBLIC_EVENT][id] == nil then
				self.tSavedData[KEY_PUBLIC_EVENT][id]={}
--				self.tSavedData[KEY_PUBLIC_EVENT][id]["id"]=game_id
				self.tSavedData[KEY_PUBLIC_EVENT][id]["name"]=e:GetName()
				self.tSavedData[KEY_PUBLIC_EVENT][id]["type"]=e:GetEventType()
				self.tSavedData[KEY_PUBLIC_EVENT][id]["rtype"]=e:GetRewardType()
				self.tSavedData[KEY_PUBLIC_EVENT][id]["zone"]={}
				self.tSavedData[KEY_PUBLIC_EVENT][id]["loc"]={}
				self.tSavedData[KEY_PUBLIC_EVENT][id]["obj"]={}
				
				local p = e:GetParentEvent()
				if p then
					self.tSavedData[KEY_PUBLIC_EVENT][id]["p"]=self:AddPublicEvent(p)
				end
			end

			--merge
			
			if e:IsActive() then
				self.tSavedData[KEY_PUBLIC_EVENT][id]["zone"][self.tZone.id]=1
				if updateRZ then
					if self.tSavedData[KEY_PUBLIC_EVENT][id]["realzone"]==nil then
						self.tSavedData[KEY_PUBLIC_EVENT][id]["realzone"]=self.tZone.id
					else
						if self.tSavedData[KEY_PUBLIC_EVENT][id]["realzone"] ~= self.tZone.id then
							self.tSavedData[KEY_PUBLIC_EVENT][id]["realzone"] = -1
						end
					end
				end
			end
			
			if self.tPlayerFaction == Unit.CodeEnumFaction.DominionPlayer then
				self.tSavedData[KEY_PUBLIC_EVENT][id]["side_d"]=true
			end
			if self.tPlayerFaction == Unit.CodeEnumFaction.ExilesPlayer then
				self.tSavedData[KEY_PUBLIC_EVENT][id]["side_e"]=true
			end
				
			for idx,upos in pairs(e:GetLocations()) do
				self.tSavedData[KEY_PUBLIC_EVENT][id]["loc"][upos.x.."/"..upos.y.."/"..upos.z]=1
			end
			for idx,obj in pairs(e:GetObjectives()) do
				local v=obj:GetCategory().."//"..obj:GetObjectiveId().."//"
				if obj:GetObjectiveType() then
					v = v .. obj:GetObjectiveType()
				end
				self.tSavedData[KEY_PUBLIC_EVENT][id]["obj"][obj:GetDescription()]=v
				--local peop=obj:GetParentObjective() id or object?
			end
		end
			
	end
	end
	return game_id
end

function Jabbithole:AddPathEpisode(e)
	if self.bInitializedProperly then
	if e then
		if self.tSavedData[KEY_PATH_EPISODE] == nil then
			self.tSavedData[KEY_PATH_EPISODE]={}
		end

		local path=PlayerPathLib.GetPlayerPathType()
		local id=path.."/"..e:GetWorldZone()
		if self.tSavedData[KEY_PATH_EPISODE][id] == nil then
			self.tSavedData[KEY_PATH_EPISODE][id]={}
--			self.tSavedData[KEY_PATH_EPISODE][id]["z"]={}
			self.tSavedData[KEY_PATH_EPISODE][id]["pm"]={}
		end
		if self.tSavedData[KEY_PATH_EPISODE][id]["rew"] == nil then
			self.tSavedData[KEY_PATH_EPISODE][id]["rew"]={}
		end
--		self.tSavedData[KEY_PATH_EPISODE][id]["z"][self.tZone.id]=1
			
		self.tSavedData[KEY_PATH_EPISODE][id]["name"]=e:GetName()
		self.tSavedData[KEY_PATH_EPISODE][id]["sum"]=e:GetSummary()

		if self.tPlayerFaction == Unit.CodeEnumFaction.DominionPlayer then
			self.tSavedData[KEY_PATH_EPISODE][id]["side_d"]=true
		end
		if self.tPlayerFaction == Unit.CodeEnumFaction.ExilesPlayer then
			self.tSavedData[KEY_PATH_EPISODE][id]["side_e"]=true
		end

		for idx,reward in pairs(e:GetRewards() or {}) do
			if reward.eType == PlayerPathLib.PathRewardType_Item and reward.itemReward then
				self.tSavedData[KEY_PATH_EPISODE][id]["rew"][self:AddItem(reward.itemReward,-1,nil,true)]=reward.nCount
			end
		end
	
		for idx,miss in pairs(e:GetMissions() or {}) do
			if miss:GetName() ~= "" then
				self:AddPathMission(path,miss)
				self.tSavedData[KEY_PATH_EPISODE][id]["pm"][miss:GetId()]=1
			end
		end
	end
	end
end

function Jabbithole:AddPathMission(path,m,z)
	if m then
		if self.tSavedData[KEY_PATH_MISSION] == nil then
			self.tSavedData[KEY_PATH_MISSION]={}
		end
		if m:GetName() ~= "" then
			local id=m:GetId()
			if self.tSavedData[KEY_PATH_MISSION][id] == nil then
				self.tSavedData[KEY_PATH_MISSION][id]={}
				self.tSavedData[KEY_PATH_MISSION][id]["p"]=path
				self.tSavedData[KEY_PATH_MISSION][id]["done"]=m:GetCompletedString()
				self.tSavedData[KEY_PATH_MISSION][id]["name"]=m:GetName()
				self.tSavedData[KEY_PATH_MISSION][id]["needed"]=m:GetNumNeeded()
				self.tSavedData[KEY_PATH_MISSION][id]["xp"]=m:GetRewardXp()
				if m:GetSpell() then
					self.tSavedData[KEY_PATH_MISSION][id]["spell"]=m:GetSpell():GetId()
				end
				self.tSavedData[KEY_PATH_MISSION][id]["type"]=m:GetType()
				self.tSavedData[KEY_PATH_MISSION][id]["stype"]=m:GetSubType()
				self.tSavedData[KEY_PATH_MISSION][id]["sum"]=m:GetSummary()
				self.tSavedData[KEY_PATH_MISSION][id]["unl"]=m:GetUnlockString()
				self.tSavedData[KEY_PATH_MISSION][id]["loc"]={}
			end

			if self.tPlayerFaction == Unit.CodeEnumFaction.DominionPlayer then
				self.tSavedData[KEY_PATH_MISSION][id]["side_d"]=true
			end
			if self.tPlayerFaction == Unit.CodeEnumFaction.ExilesPlayer then
				self.tSavedData[KEY_PATH_MISSION][id]["side_e"]=true
			end
								
--			for idx,loc in pairs(m:GetMapLocations() or {}) do
--				self.tSavedData[KEY_PATH_MISSION][id]["loc"][loc.x.."/"..loc.y.."/"..loc.z]=1
--			end
			
--			if z then
--				self.tSavedData[KEY_PATH_MISSION][id]["rz"]=self.tZone.id
--			end
		end
	end
end

function Jabbithole:AddSchematic(id,tsid)
	if tsid and id then
		if self.tSavedData[KEY_SCHEMATIC] == nil then
			self.tSavedData[KEY_SCHEMATIC]={}
		end
		
		if self.tSavedData[KEY_SCHEMATIC][tsid]==nil then
			self.tSavedData[KEY_SCHEMATIC][tsid]={}
		end
		
		local sch=CraftingLib.GetSchematicInfo(id)
		if sch then
			if self.tSavedData[KEY_SCHEMATIC][tsid][id]==nil then
				self.tSavedData[KEY_SCHEMATIC][tsid][id]={}
			end
			self.tSavedData[KEY_SCHEMATIC][tsid][id]["tier"]=sch.eTier
			self.tSavedData[KEY_SCHEMATIC][tsid][id]["autocr"]=sch.bIsAutoCraft
			self.tSavedData[KEY_SCHEMATIC][tsid][id]["autolrn"]=sch.bIsAutoLearn
			self.tSavedData[KEY_SCHEMATIC][tsid][id]["oneuse"]=sch.bIsOneUse
			self.tSavedData[KEY_SCHEMATIC][tsid][id]["univ"]=sch.bIsUniveral
			if sch.itemOutput then
				self.tSavedData[KEY_SCHEMATIC][tsid][id]["item"]=self:AddItem(sch.itemOutput,-1,nil,false,true)
			end
			self.tSavedData[KEY_SCHEMATIC][tsid][id]["xp"]=sch.nCraftXp.."/"..sch.nFailXp.."/"..sch.nLearnXp
			self.tSavedData[KEY_SCHEMATIC][tsid][id]["count"]=sch.nCreateCount
			-- .."/"..sch.nCritCount removed in reloaded
			self.tSavedData[KEY_SCHEMATIC][tsid][id]["parent"]=sch.nParentSchematicId
			-- workaround attempt for missing parent schematics
			-- possible endless loop?
			if sch.nParentSchematicId ~= 0 and self.tSavedData[KEY_SCHEMATIC][tsid][sch.nParentSchematicId] == nil then
				self:AddSchematic(sch.nParentSchematicId,tsid)
			end
			self.tSavedData[KEY_SCHEMATIC][tsid][id]["maxadd"]=sch.nMaxAdditives
			self.tSavedData[KEY_SCHEMATIC][tsid][id]["maxcat"]=sch.nMaxCatalysts
			self.tSavedData[KEY_SCHEMATIC][tsid][id]["name"]=sch.strName
			
			if sch.arMaterials and #sch.arMaterials > 0 then
				self.tSavedData[KEY_SCHEMATIC][tsid][id]["mats"]={}
				for idx,mat in pairs(sch.arMaterials or {}) do
					if mat.itemMaterial then
						self.tSavedData[KEY_SCHEMATIC][tsid][id]["mats"][self:AddItem(mat.itemMaterial,-1,nil,false)]=mat.nNeeded
					end
				end
			end
			if sch.tSubRecipes and #sch.tSubRecipes > 0 then
				self.tSavedData[KEY_SCHEMATIC][tsid][id]["subs"]={}
				for idx,subrec in pairs(sch.tSubRecipes or {}) do
					self:AddSchematic(subrec.nSchematicId,tsid)
					self.tSavedData[KEY_SCHEMATIC][tsid][id]["subs"][subrec.nSchematicId]=1
				end
			end
			if sch.arStats and #sch.arStats > 0 then
				self.tSavedData[KEY_SCHEMATIC][tsid][id]["stats"]={}
				for idx,stat in pairs(sch.arStats or {}) do
					self.tSavedData[KEY_SCHEMATIC][tsid][id]["stats"][idx]={}
					self.tSavedData[KEY_SCHEMATIC][tsid][id]["stats"][idx]["ist"]=stat.eItemStatType
					self.tSavedData[KEY_SCHEMATIC][tsid][id]["stats"][idx]["rat"]=stat.nRatio
					if stat.arCraftingGroups and #stat.arCraftingGroups > 0 then
						self.tSavedData[KEY_SCHEMATIC][tsid][id]["stats"][idx]["cgs"]=""
						for idx2,cg in pairs(stat.arCraftingGroups or {}) do
							if idx2>1 then
								self.tSavedData[KEY_SCHEMATIC][tsid][id]["stats"][idx]["cgs"]=self.tSavedData[KEY_SCHEMATIC][tsid][id]["stats"][idx]["cgs"]..","
							end
							self.tSavedData[KEY_SCHEMATIC][tsid][id]["stats"][idx]["cgs"]=self.tSavedData[KEY_SCHEMATIC][tsid][id]["stats"][idx]["cgs"]..cg
						end
					end
				end
			end
			--[[ not used in reloaded anymore
			if sch.tSockets and #sch.tSockets > 0 then
				self.tSavedData[KEY_SCHEMATIC][tsid][id]["sock"]={}
				for idx,sock in pairs(sch.tSockets or {}) do
					local sd=""
					if sock.bIsChangeable then
						sd="+"
					else
						sd="-"
					end
					sd=sd.."/"..sock.eSocketType.."/"..sock.fRatio.."/"..sock.nParent.."/"
					if sock.itemDefaultChip ~= nil then
						sd=sd..self:AddItem(sock.itemDefaultChip)
					end
					self.tSavedData[KEY_SCHEMATIC][tsid][id]["sock"][idx]=sd
				end
				-- not used in reloaded anymore, simply points to the crafted item
				local preview = CraftingLib.GetPreviewInfo(id)
				if preview then
					self.tSavedData[KEY_SCHEMATIC][tsid][id]["prev"]={}
					for idx,sock in pairs(preview.tSockets or {}) do
						self.tSavedData[KEY_SCHEMATIC][tsid][id]["prev"][idx]=sock.fMultiplier.."/"..sock.nPower.."/"..sock.nThresholdCount
					end
				end
				
			end
			]]
		end
	end
end

function Jabbithole:AddTradeskillTalents(id)
	if id then
		if self.tSavedData[KEY_TRADESKILL_TALENT] == nil then
			self.tSavedData[KEY_TRADESKILL_TALENT]={}
		end
		
		if self.tSavedData[KEY_TRADESKILL_TALENT][id]==nil then
			self.tSavedData[KEY_TRADESKILL_TALENT][id]={}
		end
			
		local talents=CraftingLib.GetTradeskillTalents(id)
		if talents and #talents>0 then
			for idx,tier in pairs(talents) do
				self.tSavedData[KEY_TRADESKILL_TALENT][id][idx]={}
				self.tSavedData[KEY_TRADESKILL_TALENT][id][idx]["p"]=tier.nPointsRequired
				self.tSavedData[KEY_TRADESKILL_TALENT][id][idx]["t"]={}
				for idx2,talent in pairs(tier.tTalents or {}) do
					self.tSavedData[KEY_TRADESKILL_TALENT][id][idx]["t"][talent.nTalentId]={}
					self.tSavedData[KEY_TRADESKILL_TALENT][id][idx]["t"][talent.nTalentId]["n"]=talent.strName
					self.tSavedData[KEY_TRADESKILL_TALENT][id][idx]["t"][talent.nTalentId]["i"]=talent.strIcon
					self.tSavedData[KEY_TRADESKILL_TALENT][id][idx]["t"][talent.nTalentId]["t"]=talent.strTooltip
				end
			end
		end
	end
end

function Jabbithole:AddTradeskill(id,ts)
	if ts and id and ts.bIsActive then
		if self.tSavedData[KEY_TRADESKILL] == nil then
			self.tSavedData[KEY_TRADESKILL]={}
		end
		
		if self.tSavedData[KEY_TRADESKILL][id]==nil then
			self.tSavedData[KEY_TRADESKILL][id]={}
		end

		self.tSavedData[KEY_TRADESKILL][id]["hobby"]=ts.bIsHobby
		self.tSavedData[KEY_TRADESKILL][id]["harv"]=ts.bIsHarvesting
		self.tSavedData[KEY_TRADESKILL][id]["tmax"]=ts.nTierMax
		self.tSavedData[KEY_TRADESKILL][id]["desc"]=ts.strDescription
		self.tSavedData[KEY_TRADESKILL][id]["name"]=ts.strName
		-- TODO:
		-- tAxisNames??
		-- tTalentTiers??
		-- tXpTiers??
		
		local grps=AchievementsLib.GetTradeskillAchievementCategoryTree(id)
		if grps ~= nil then
			self.tSavedData[KEY_TRADESKILL][id]["agrpid"]=grps.nGroupId
			self.tSavedData[KEY_TRADESKILL][id]["agrpname"]=grps.strGroupName

			local side=nil
			if self.tPlayerFaction == Unit.CodeEnumFaction.DominionPlayer then
				side="dom"
			end
			if self.tPlayerFaction == Unit.CodeEnumFaction.ExilesPlayer then
				side="exi"
			end
			
			if side then
				local tiersidekey = "tiers"..side
				self.tSavedData[KEY_TRADESKILL][id][tiersidekey]={}
				
				for idx,tier in pairs(grps.tSubGroups or {}) do
					self.tSavedData[KEY_TRADESKILL][id][tiersidekey][idx]={}
					self.tSavedData[KEY_TRADESKILL][id][tiersidekey][idx]["agrpid"]=tier.nSubGroupId
					self.tSavedData[KEY_TRADESKILL][id][tiersidekey][idx]["agrpname"]=tier.strSubGroupName
					self.tSavedData[KEY_TRADESKILL][id][tiersidekey][idx]["tree"]={}
					local tree=AchievementsLib.GetTradeskillAchievementLayout(tier.nSubGroupId)
					for idx2,treenode in pairs(tree or {}) do
					
						--Print("XX: "..id.." "..tier.nSubGroupId.." "..idx2)
					
						self.tSavedData[KEY_TRADESKILL][id][tiersidekey][idx]["tree"][idx2]={}
						self.tSavedData[KEY_TRADESKILL][id][tiersidekey][idx]["tree"][idx2]["id"]=treenode:GetId()
						self.tSavedData[KEY_TRADESKILL][id][tiersidekey][idx]["tree"][idx2]["name"]=nlFix(treenode:GetName())
						self.tSavedData[KEY_TRADESKILL][id][tiersidekey][idx]["tree"][idx2]["desc"]=nlFix(treenode:GetDescription())
						self.tSavedData[KEY_TRADESKILL][id][tiersidekey][idx]["tree"][idx2]["num"]=treenode:GetNumNeeded()
						self.tSavedData[KEY_TRADESKILL][id][tiersidekey][idx]["tree"][idx2]["pts"]=treenode:GetPoints()
						self.tSavedData[KEY_TRADESKILL][id][tiersidekey][idx]["tree"][idx2]["ptxt"]=treenode:GetProgressText()
						local lo=treenode:GetTradeskillLayout()
						self.tSavedData[KEY_TRADESKILL][id][tiersidekey][idx]["tree"][idx2]["lox"]=lo.x
						self.tSavedData[KEY_TRADESKILL][id][tiersidekey][idx]["tree"][idx2]["loy"]=lo.y
						self.tSavedData[KEY_TRADESKILL][id][tiersidekey][idx]["tree"][idx2]["lop"]={}
						for idx3,par in pairs(lo.arParents or {}) do
							self.tSavedData[KEY_TRADESKILL][id][tiersidekey][idx]["tree"][idx2]["lop"][par.x.."/"..par.y]=1
						end
						local bo=treenode:GetTradeskillRewards()
						if bo ~= nil then
							self.tSavedData[KEY_TRADESKILL][id][tiersidekey][idx]["tree"][idx2]["rwtp"]=bo.nTalentPoints
							self.tSavedData[KEY_TRADESKILL][id][tiersidekey][idx]["tree"][idx2]["rwbon"]={}
							for idx3,bon in pairs(bo.arBonuses or {}) do
								self.tSavedData[KEY_TRADESKILL][id][tiersidekey][idx]["tree"][idx2]["rwbon"][idx3]=bon.strIcon.."//"..bon.strName.."//"..bon.strTooltip
							end
							self.tSavedData[KEY_TRADESKILL][id][tiersidekey][idx]["tree"][idx2]["rwsch"]={}
							for idx3,sch in pairs(bo.arSchematics or {}) do
								self:AddSchematic(sch.idSchematic, id)
								self.tSavedData[KEY_TRADESKILL][id][tiersidekey][idx]["tree"][idx2]["rwsch"][idx3]=sch.idSchematic
							end
						end
						local cl=treenode:GetChecklistItems()
						self.tSavedData[KEY_TRADESKILL][id][tiersidekey][idx]["tree"][idx2]["cl"]={}
						for idx3,cli in pairs(cl or {}) do
							if cli.idSchematic ~= nil then
								self:AddSchematic(cli.idSchematic, id)
								self.tSavedData[KEY_TRADESKILL][id][tiersidekey][idx]["tree"][idx2]["cl"][idx3]=cli.idSchematic.."//"..cli.strChecklistEntry
							else
								self.tSavedData[KEY_TRADESKILL][id][tiersidekey][idx]["tree"][idx2]["cl"][idx3]="//"..cli.strChecklistEntry
							end
						end
					end
				end
			end
					
		end
		
		local schs=nil
		if self.tSavedData[KEY_SCHEMATIC]==nil or self.tSavedData[KEY_SCHEMATIC][id]==nil then
			schs=CraftingLib.GetSchematicList(id,nil,nil,true)
		else
			schs=CraftingLib.GetSchematicList(id)
		end
		for idx,sch in pairs(schs or {}) do
			self:AddSchematic(sch.nSchematicId, id)
		end
		
		self:AddTradeskillTalents(id)
		Apollo.StartTimer("TradeskillScanner")
	end
end

function Jabbithole:OnTradeskillTimer()
--	Print("TST "..self.tradeskillIndex)
	if self.tradeskillIndex and self.tradeskillsToScan and self.tradeskillsToScan[self.tradeskillIndex] then
		local ts=self.tradeskillsToScan[self.tradeskillIndex]
		local tsi=CraftingLib.GetTradeskillInfo(ts.eId)
		self:AddTradeskill(ts.eId,tsi)
		self.tradeskillIndex = self.tradeskillIndex+1
	else
		lastTradeskillSaveTime=os.time()
	end
end

function Jabbithole:AddTradeskills()
--	Print("TSTS "..self.tSavedData["lasttssave"])

	Apollo.RegisterTimerHandler("AchievementScanner", "OnAchievementTimer", self)
	Apollo.CreateTimer("AchievementScanner", 5, false)
	Apollo.StartTimer("AchievementScanner")

	if lastTradeskillSaveTime+79200<os.time() then
		self.tradeskillIndex=1
		self.tradeskillsToScan=CraftingLib.GetKnownTradeskills()
		Apollo.RegisterTimerHandler("TradeskillScanner", "OnTradeskillTimer", self)
		Apollo.CreateTimer("TradeskillScanner", 10, false)
		Apollo.StartTimer("TradeskillScanner")
	end
end

function Jabbithole:AddReputation(faction, label, group, side)
	if self.tSavedData[KEY_FACTION] == nil then
		self.tSavedData[KEY_FACTION]={}
	end
	
	local key = faction.."//"..group.."//"..label
	local xside = self.tSavedData[KEY_FACTION][key]
	if xside == nil then
		self.tSavedData[KEY_FACTION][key] = side
	else
		if xside + side == 1 then
			self.tSavedData[KEY_FACTION][key] = 2
		end
	end
end

function Jabbithole:AddReputations()
	local r=GameLib.GetReputationInfo()
	local groups={}
	local labels={}

	local side=-1	
	if self.tPlayerFaction == Unit.CodeEnumFaction.DominionPlayer then
		side=0
	end
	if self.tPlayerFaction == Unit.CodeEnumFaction.ExilesPlayer then
		side=1
	end

	if side ~= -1 then
		for idx,rep in pairs(r or {}) do
			if rep.bIsLabel == true and rep.strParent ~= "" then
				labels[rep.strName]=rep.strParent
			end
			if rep.bIsLabel == false and rep.strParent == "" then
				groups[rep.strName]=rep.strName
				self:AddReputation(rep.strName, "", "", side)
			end
		end
		for idx,rep in pairs(r or {}) do
			if rep.bIsLabel == false and rep.strParent ~= "" then
				if labels[rep.strParent] then
					if groups[labels[rep.strParent]] then
						self:AddReputation(rep.strName, rep.strParent, labels[rep.strParent], side)
					end
				else
					if groups[rep.strParent] then
						self:AddReputation(rep.strName, "", rep.strParent, side)
					else
						self:AddReputation(rep.strName, "", rep.strParent, side)
					end
				end
			end
		end
	end
end

function Jabbithole:AddHousingDecorType(dt)
	if self.tSavedData[KEY_HOUSE_DECORTYPE] == nil then
		self.tSavedData[KEY_HOUSE_DECORTYPE]={}
	end
	if dt then
		local id = dt.nId
		if self.tSavedData[KEY_HOUSE_DECORTYPE][id] == nil then
			self.tSavedData[KEY_HOUSE_DECORTYPE][id]={}
		end
		self.tSavedData[KEY_HOUSE_DECORTYPE][id]["id"] = id
		self.tSavedData[KEY_HOUSE_DECORTYPE][id]["name"] = dt.strName
	end
end

function Jabbithole:AddHousingDecor(deco)
	if self.tSavedData[KEY_HOUSE_DECOR] == nil then
		self.tSavedData[KEY_HOUSE_DECOR]={}
	end
	if deco then
		local id = deco.nId
		if self.tSavedData[KEY_HOUSE_DECOR][id] == nil then
			self.tSavedData[KEY_HOUSE_DECOR][id]={}
		end
		self.tSavedData[KEY_HOUSE_DECOR][id]["id"] = id
		self.tSavedData[KEY_HOUSE_DECOR][id]["name"] = deco.strName
		self.tSavedData[KEY_HOUSE_DECOR][id]["curr"] = deco.eCurrencyType
		self.tSavedData[KEY_HOUSE_DECOR][id]["etype"] = deco.eDecorType
		self.tSavedData[KEY_HOUSE_DECOR][id]["gcurr"] = deco.eGroupCurrencyType
		self.tSavedData[KEY_HOUSE_DECOR][id]["cost"] = deco.nCost
		self.tSavedData[KEY_HOUSE_DECOR][id]["ht"] = deco.eHookType
		self.tSavedData[KEY_HOUSE_DECOR][id]["fl"] = deco.nFlags
		self.tSavedData[KEY_HOUSE_DECOR][id]["htaid"] = deco.nHookAssetId
		if deco.splBuff then
			self:AddSpell(deco.splBuff)
			self.tSavedData[KEY_HOUSE_DECOR][id]["spl"] = deco.splBuff:GetId()
		end
	end
end

--[[ obsolete since reloaded
function Jabbithole:AddHousingDecorCrate(deco)
	if self.tSavedData[KEY_HOUSE_DECOR] == nil then
		self.tSavedData[KEY_HOUSE_DECOR]={}
	end
	if deco then
		local id = deco.nId
		if self.tSavedData[KEY_HOUSE_DECOR][id] == nil then
			self.tSavedData[KEY_HOUSE_DECOR][id]={}
		end
		self.tSavedData[KEY_HOUSE_DECOR][id]["id"] = id
		self.tSavedData[KEY_HOUSE_DECOR][id]["name"] = deco.strName
		self.tSavedData[KEY_HOUSE_DECOR][id]["etype"] = deco.eDecorType
		if deco.splBuff then
			self:AddSpell(deco.splBuff)
			self.tSavedData[KEY_HOUSE_DECOR][id]["spl"] = deco.splBuff:GetId()
		end
	end
end
]]

function Jabbithole:AddHousingPlot(plot, parentId)
	if self.tSavedData[KEY_HOUSE_PLOTS] == nil then
		self.tSavedData[KEY_HOUSE_PLOTS]={}
	end
	if plot then
		local id = plot.nId
		if self.tSavedData[KEY_HOUSE_PLOTS][id] == nil then
			self.tSavedData[KEY_HOUSE_PLOTS][id]={}
		end
-- redundant
--		self.tSavedData[KEY_HOUSE_PLOTS][id]["id"] = id
		self.tSavedData[KEY_HOUSE_PLOTS][id]["etype"] = plot.eType
		self.tSavedData[KEY_HOUSE_PLOTS][id]["parent"] = parentId
		self.tSavedData[KEY_HOUSE_PLOTS][id]["cost"] = plot.nCost
		self.tSavedData[KEY_HOUSE_PLOTS][id]["name"] = plot.strName
		self.tSavedData[KEY_HOUSE_PLOTS][id]["desc"] = plot.strTooltip
		self.tSavedData[KEY_HOUSE_PLOTS][id]["shots"] = {}
		for idx, shot in pairs(plot.tScreenshots or {}) do
			if shot.strSprite then
				self.tSavedData[KEY_HOUSE_PLOTS][id]["shots"][idx]=shot.strSprite
			end
		end
		self.tSavedData[KEY_HOUSE_PLOTS][id]["prereq"] = {}
		for idx, pre in pairs(plot.tPrerequisites or {}) do
			if pre.strTooltip then
				self.tSavedData[KEY_HOUSE_PLOTS][id]["prereq"][idx]=pre.strTooltip 
			end
		end
		self.tSavedData[KEY_HOUSE_PLOTS][id]["cost"] = {}
		for idx, cost in pairs(plot.tCostRequirements or {}) do
			self.tSavedData[KEY_HOUSE_PLOTS][id]["cost"][idx]={}
			self.tSavedData[KEY_HOUSE_PLOTS][id]["cost"][idx]["etype"]=cost.eType
			self.tSavedData[KEY_HOUSE_PLOTS][id]["cost"][idx]["amo"]=cost.nRequiredCost
			if cost.itemCostReq then
				self.tSavedData[KEY_HOUSE_PLOTS][id]["cost"][idx]["item"]=cost.itemCostReq:GetItemId()
				self:AddItem(cost.itemCostReq,-1,nil,true)
			end
		end
	end
end

function Jabbithole:SaveHousingStuff()
	if HousingLib.IsHousingWorld() and HousingLib.IsOnMyResidence() and not HousingLib.GetResidence():IsWarplotResidence() then
		HousingLib.RequestVendorList()
		local plots = HousingLib.GetResidence():GetPlotCount()
		for idx=1, plots do
			plot = HousingLib.GetPlot(idx)
			if plot and plot:HasUpgrade() then
				local up = plot:GetPlugItemUpgrade()
				if up then
					self:AddHousingPlot(up, plot:GetPlugItemId())
				end
			end
		end
		
		for idx, dt in pairs(HousingLib.GetDecorTypeList() or {}) do
			self:AddHousingDecorType(dt)
		end

		for idx, dc in pairs(HousingLib.GetDecorCatalogList() or {}) do
			self:AddHousingDecor(dc)
		end
		for idx, dc in pairs(HousingLib.GetResidence():GetDecorCrateList() or {}) do
			self:AddHousingDecor(dc)
		end
	end	
end

function Jabbithole:AddAchievementGroup(id,name,parent)
	if self.tSavedData[KEY_ACHI_GROUP] == nil then
		self.tSavedData[KEY_ACHI_GROUP]={}
	end
	if self.tSavedData[KEY_ACHI_GROUP][id] == nil then
		self.tSavedData[KEY_ACHI_GROUP][id]={}
	end
	self.tSavedData[KEY_ACHI_GROUP][id]["name"] = name
	self.tSavedData[KEY_ACHI_GROUP][id]["parent"] = parent
end

function Jabbithole:AddAchievement(parent, ach, isguild, faction, tier, aparent)
	if self.tSavedData[KEY_ACHI] == nil then
		self.tSavedData[KEY_ACHI]={}
	end
	local id = ach:GetId()
	if self.tSavedData[KEY_ACHI][id] == nil then
		self.tSavedData[KEY_ACHI][id]={}
	end
	self.tSavedData[KEY_ACHI][id]["name"] = ach:GetName()
	self.tSavedData[KEY_ACHI][id]["parent"] = parent
	self.tSavedData[KEY_ACHI][id]["aparent"] = aparent
	self.tSavedData[KEY_ACHI][id]["tier"] = tier
	self.tSavedData[KEY_ACHI][id]["g"] = isguild
	if faction == Unit.CodeEnumFaction.DominionPlayer then
		self.tSavedData[KEY_ACHI][id]["side_d"]=true
	end
	if faction == Unit.CodeEnumFaction.ExilesPlayer then
		self.tSavedData[KEY_ACHI][id]["side_e"]=true
	end
	self.tSavedData[KEY_ACHI][id]["pts"] = ach:GetPoints()
	self.tSavedData[KEY_ACHI][id]["num"] = ach:GetNumNeeded()
	self.tSavedData[KEY_ACHI][id]["desc"] = ach:GetDescription()
	self.tSavedData[KEY_ACHI][id]["text"] = ach:GetProgressText()
	self.tSavedData[KEY_ACHI][id]["wz"] = ach:GetWorldZoneId()
	if ach:GetRewards() and ach:GetRewards().strTitle then
		self:AddTitle(ach:GetRewards().strTitle:GetTitle(),ach:GetRewards().strTitle:GetCategory(),ach:GetRewards().strTitle:GetForUnit(),id,ach:GetRewards().strTitle:GetSpell())
		self.tSavedData[KEY_ACHI][id]["rewt"]=ach:GetRewards().strTitle:GetTitle()
	end
	self.tSavedData[KEY_ACHI][id]["cl"] = {}
	for key, clitem in pairs(ach:GetChecklistItems() or {}) do
		self.tSavedData[KEY_ACHI][id]["cl"][#self.tSavedData[KEY_ACHI][id]["cl"]+1] = clitem.strChecklistEntry
	end	
end

function Jabbithole:AddAchievements(parent, isguild, faction)
	for key, achCurr in pairs(AchievementsLib.GetAchievementsForCategory(parent,false,isguild) or {}) do
		if achCurr:GetParentTier()==nil and achCurr:GetChildTier()==nil then
			self:AddAchievement(parent, achCurr, isguild, faction, 1, -1)
		else
			if achCurr:GetParentTier()==nil and achCurr:GetChildTier()~=nil then
			
				local achTopMostParent = achCurr
				local nSafetyCount = 0
				local lastId = -1
				
				while achTopMostParent and nSafetyCount < 99 do
					nSafetyCount = nSafetyCount + 1
					self:AddAchievement(parent, achTopMostParent, isguild, faction, nSafetyCount, lastId)
					lastId = achTopMostParent:GetId()
					achTopMostParent = achTopMostParent:GetChildTier()
				end
			end
		end
	end
end

function Jabbithole:AddAchievementGroups(isguild)
	local faction = GameLib.GetPlayerUnit():GetFaction()
	local tCategoryTree = AchievementsLib.GetAchievementCategoryTree(isguild)

	for key, tTopLevel in pairs(tCategoryTree) do
		self:AddAchievementGroup(tTopLevel.nCategoryId,tTopLevel.strCategoryName,-1)
		self:AddAchievements(tTopLevel.nCategoryId, isguild, faction)
		for key2, tMiddleLevel in pairs(tTopLevel.tGroups) do
			self:AddAchievementGroup(tMiddleLevel.nGroupId,tMiddleLevel.strGroupName,tTopLevel.nCategoryId)
			self:AddAchievements(tMiddleLevel.nGroupId, isguild, faction)
			for key3, tLowLevel in pairs(tMiddleLevel.tSubGroups) do
				self:AddAchievementGroup(tLowLevel.nSubGroupId,tLowLevel.strSubGroupName,tMiddleLevel.nGroupId)
				self:AddAchievements(tLowLevel.nSubGroupId, isguild, faction)
			end
		end
	end
end

function Jabbithole:OnAchievementTimer()
	Apollo.RegisterTimerHandler("ChallengesScanner", "OnChallengesTimer", self)
	Apollo.CreateTimer("ChallengesScanner", 3, false)
	Apollo.StartTimer("ChallengesScanner")

	if GameLib.GetPlayerUnit() and GameLib.GetPlayerUnit():GetFaction() then
		Apollo.StopTimer("AchievementScanner")
		self:AddAchievementGroups(false)
		self:AddAchievementGroups(true)
	else
		Apollo.StartTimer("AchievementScanner")	
	end
end

function Jabbithole:AddContract(contract)
	if self.tSavedData[KEY_CONTRACTS] == nil then
		self.tSavedData[KEY_CONTRACTS]={}
	end
	local q=contract:GetQuest()
	if q then
		local id = contract:GetId()
		if self.tSavedData[KEY_CONTRACTS][id] == nil then
			self.tSavedData[KEY_CONTRACTS][id]={}
		end
		
		self.tSavedData[KEY_CONTRACTS][id]["id"] = id
		self:AddQuest(q,nil,nil,true,false)
		self.tSavedData[KEY_CONTRACTS][id]["quest"] = q:GetId()
	
		if self.tPlayerFaction == Unit.CodeEnumFaction.DominionPlayer then
			self.tSavedData[KEY_CONTRACTS][id]["side_d"]=true
		end
		if self.tPlayerFaction == Unit.CodeEnumFaction.ExilesPlayer then
			self.tSavedData[KEY_CONTRACTS][id]["side_e"]=true
		end

		self.tSavedData[KEY_CONTRACTS][id]["q"] = contract:GetQuality()
		self.tSavedData[KEY_CONTRACTS][id]["rtt"] = contract:GetRewardTrackType()
		self.tSavedData[KEY_CONTRACTS][id]["xp"] = contract:GetRewardTrackValue()
		self.tSavedData[KEY_CONTRACTS][id]["typ"] = contract:GetType()
	end
	
end

function Jabbithole:AddContracts()
	for idx, arr in pairs(ContractsLib.GetPeriodicContracts() or {}) do
		for idx2, contract in pairs(arr or {}) do
			self:AddContract(contract)
		end
	end
end

function Jabbithole:AddContractReward(reward, tier, itemchoice)
	if self.tSavedData[KEY_CONTRACTREWARDS] == nil then
		self.tSavedData[KEY_CONTRACTREWARDS]={}
	end
	if reward and tier and itemchoice and itemchoice.itemReward then
		id=reward:GetType().."."..reward:GetId().."."..tier.idReward.."."..itemchoice.itemReward:GetItemId()
		if self.tSavedData[KEY_CONTRACTREWARDS][id] == nil then
			self.tSavedData[KEY_CONTRACTREWARDS][id]={}
		end
		self.tSavedData[KEY_CONTRACTREWARDS][id]["trid"]=reward:GetId()
		self.tSavedData[KEY_CONTRACTREWARDS][id]["trass"]=reward:GetImageAssetPath()
		self.tSavedData[KEY_CONTRACTREWARDS][id]["trtyp"]=reward:GetType()
		self.tSavedData[KEY_CONTRACTREWARDS][id]["rewcas"]=tier.bIsCasque
		self.tSavedData[KEY_CONTRACTREWARDS][id]["rewid"]=tier.idReward
		self.tSavedData[KEY_CONTRACTREWARDS][id]["rewcost"]=tier.nCost
		self.tSavedData[KEY_CONTRACTREWARDS][id]["rewidx"]=tier.nRewardIdx
		self.tSavedData[KEY_CONTRACTREWARDS][id]["item"]=itemchoice.itemReward:GetItemId()
		self.tSavedData[KEY_CONTRACTREWARDS][id]["itemamo"]=itemchoice.nRewardCount
		self.tSavedData[KEY_CONTRACTREWARDS][id]["itemcidx"]=itemchoice.nChoiceIdx
		if self.tPlayerFaction == Unit.CodeEnumFaction.DominionPlayer then
			self.tSavedData[KEY_CONTRACTREWARDS][id]["side_d"]=true
		end
		if self.tPlayerFaction == Unit.CodeEnumFaction.ExilesPlayer then
			self.tSavedData[KEY_CONTRACTREWARDS][id]["side_e"]=true
		end
	end
end

function Jabbithole:AddContractRewards()
	for idx, rew in pairs(RewardTrackLib.GetAllRewardTracks() or {}) do
		if rew:GetType() == RewardTrackLib.CodeEnumRewardTrackType.ContractPvE or rew:GetType() == RewardTrackLib.CodeEnumRewardTrackType.ContractPvP then
			for idx2, tier in pairs(rew:GetAllRewards() or {}) do
				for idx3, itemch in pairs(tier.tRewardChoices or {}) do
					self:AddContractReward(rew, tier, itemch)
				end
			end
		end
	end
end


function Jabbithole:AddCharacter()
end

function Jabbithole:AddCharacterLater()
	local player = GameLib.GetPlayerUnit()
	if self.accountVars.saveArmory and player and player:GetLevel()>9 then
		if self.tSavedData[KEY_CHARACTERS] == nil then
			self.tSavedData[KEY_CHARACTERS]={}
		end
		
		local id = player:GetName() ..'@'.. GameLib.GetRealmName()
		
		local char=self.tSavedData[KEY_CHARACTERS][id] or {}

		char.fullname=player:GetTitleOrName()
		char.faction=player:GetFaction()
		char.class=player:GetClassId()
		char.gender=player:GetGender()
		char.level=player:GetLevel()
		char.path=player:GetPlayerPathType()
		char.pathlevel=PlayerPathLib.GetPathLevel()
		char.race=player:GetRaceId()
		
		for idx, guild in pairs(GuildLib:GetGuilds() or {}) do
			if guild:GetType()==GuildLib.GuildType_Guild then
				char.guild=guild:GetName()
			end
		end
		
		char.items={}
		char.runes={}
		char.imbue={}
		for idx, item in pairs(player:GetEquippedItems() or {}) do
			char.items[idx]=self:AddItem(item,-1,nil,true)
			char.runes[idx]={}
			local info=item:GetDetailedInfo()
			if info.tPrimary and info.tPrimary.tRunes then
				for ridx,sigil in pairs(info.tPrimary.tRunes.arRuneSlots or {}) do
					if sigil.itemSigil then
						char.runes[idx][ridx]=self:AddItem(sigil.itemSigil).."/"..sigil.eProperty.."/"..sigil.nValue
					else
						char.runes[idx][ridx]=""
					end
				end
			end
			if info.tPrimary and info.tPrimary.arImbuements then
				local imbComplete=0
				for ridx,imbuement in pairs(info.tPrimary.arImbuements or {}) do
					if imbuement.bComplete then
						imbComplete = imbComplete + 1
					end
				end
				char.imbue[""..idx]=imbComplete
			end
		end
		
		char.stats={}
		for key, stat in pairs(player:GetUnitProperties() or {}) do
			char.stats[stat.idProperty]=stat.fBase.."/"..stat.fValue
		end
		
		char.sstats={}
		char.sstats.StrikethroughChance= (math.floor((player:GetStrikethroughChance() + 0.000005) * 10000) / 100)
		char.sstats.CriticalHitChance= (math.floor((player:GetCritChance() + 0.000005) * 10000) / 100)
		char.sstats.CriticalHitSeverity= (math.floor((player:GetCritSeverity() + 0.000005) * 10000) / 100)
		char.sstats.ArmorPierce= (math.floor((player:GetIgnoreArmorBase() + 0.000005) * 10000) / 100)
		char.sstats.ShieldPierce= (math.floor((player:GetIgnoreShieldBase() + 0.000005) * 10000) / 100)
		char.sstats.Lifesteal= (math.floor((player:GetBaseLifesteal() + 0.000005) * 10000) / 100)
		char.sstats.CooldownReduction= (-1 * math.floor((player:GetCooldownReductionModifier() + 0.000005 - 1) * 10000) / 100)
		char.sstats.ShieldRegenRate= (math.floor((player:GetShieldRegenPct() + 0.000005) * 10000) / 100)
		char.sstats.ShieldRebootTime= player:GetShieldRebootTime() / 1000
		char.sstats.PhysicalResistance= (math.floor((player:GetPhysicalMitigation() + 0.000005) * 10000) / 100)
		char.sstats.TechnologyResistance= (math.floor((player:GetTechMitigation() + 0.000005) * 10000) / 100)
		char.sstats.MagicResistance= (math.floor((player:GetMagicMitigation() + 0.000005) * 10000) / 100)
		char.sstats.DeflectChance= (math.floor((player:GetDeflectChance() + 0.000005) * 10000) / 100)
		char.sstats.DeflectCriticalHitChance= (math.floor((player:GetDeflectCritChance() + 0.000005) * 10000) / 100)
		char.sstats.CCResilience= (math.floor((math.abs(player:GetCCDurationModifier() -1) + 0.000005) * 10000) / 100)
		char.sstats.FocusRecoveryRate= player:GetManaRegenInCombat() * 2
		char.sstats.FocusCostReduction= (math.floor((math.abs(player:GetManaCostModifier() -1) + 0.000005) * 10000) / 100)
		char.sstats.PvPOffense= (math.floor((player:GetPvPOffensePercent() + 0.000005) * 10000) / 100)
		char.sstats.PvPDefense= (math.floor((player:GetPvPDefensePercent() + 0.000005) * 10000) / 100)
		
		char.rep={}
		for idx, rep in pairs(GameLib.GetReputationInfo() or {}) do
			if rep.nCurrent then
				char.rep[rep.strName]=rep.nCurrent
			end
		end
		
		char.build={}
		local abs=AbilityBook.GetAbilitiesList()
		for idx, sa in pairs(ActionSetLib.GetCurrentActionSet() or {}) do
			if idx<9 and sa~=0 then
				for idx2,ab in pairs(abs or {}) do
					if ab.nId == sa then
						char.build[idx]=sa.."/"..ab.nCurrentTier
					end
				end
			end
		end
		
		char.amps=""
		for idx, amp in pairs(AbilityBook.GetEldanAugmentationData(AbilityBook.GetCurrentSpec()).tAugments or {}) do
			if amp.eEldanAvailability == AbilityBook.CodeEnumEldanAvailability.Activated then
				if char.amps~="" then
					char.amps=char.amps..","
				end
				char.amps=char.amps..amp.nId
			end
		end
		
		char.achis={}
		for idx, achi in pairs(AchievementsLib.GetAchievements() or {}) do
			if achi:IsComplete() then
				char.achis[""..achi:GetId()]=achi:GetDateCompleted()
			end
		end
		
		char.gs=GameLib.GetGearScore()
		char.ap=AchievementsLib.GetAchievementPoints()
		char.tabp=GameLib.GetTotalAbilityPoints()
		char.tamp=AbilityBook.GetTotalPower()
		
		--events
	
		self.tSavedData[KEY_CHARACTERS][id]=char
	end
end

-----------------------------------------------------------------------------------------------
-- Jabbithole Functions
-----------------------------------------------------------------------------------------------
-- Define general functions here


function Jabbithole:OnCharacterCreated()
--	Print("Jabbithole:OnCharacterCreated()")
	self:SetupInternals()
	Apollo.StartTimer("DelayedInitTimer")
end 

function Jabbithole:OnPlayerEquippedItemChanged(nEquippedSlot, uNewItem, uOldItem)
	self:SetupInternals()
	if uNewItem then
		self:AddItem(uNewItem,-1,nil,true)
	end
end

function Jabbithole:OnItemModified(uNewItem)
	self:SetupInternals()
	if uNewItem then
		self:AddItem(uNewItem,-1,nil,true)
	end
end

function Jabbithole:OnLootedItem(item, nCount)
	self:SetupInternals()
	if item and nCount>0 then
		local itemid=self:MakeItemId(item)
		
		if droppedItems[item:GetName()] then
			droppedItems[item:GetName()]=nil
			justLootedCache[itemid]=true
		else
			--everything else that is not a drop
			self:AddItem(item,-1,nil,true)
			
			local timeKey=math.floor(GameLib.GetGameTime()*10)
			if containerCache[timeKey] == nil then
				containerCache[timeKey] = {}
			end
			if salvageCache[timeKey] == nil then
				salvageCache[timeKey] = {}
			end
			
			containerCache[timeKey][#containerCache[timeKey]+1]=itemid
			salvageCache[timeKey][#salvageCache[timeKey]+1]=itemid
			
			if reverseContainerCache[timeKey] ~= nil then
				if self.tSavedData[KEY_ITEMS][reverseContainerCache[timeKey]]["cont"] == nil then
					self.tSavedData[KEY_ITEMS][reverseContainerCache[timeKey]]["cont"]={}
				end
				if self.tSavedData[KEY_ITEMS][reverseContainerCache[timeKey]]["cont"][itemid] == nil then
					self.tSavedData[KEY_ITEMS][reverseContainerCache[timeKey]]["cont"][itemid]=1
				else
					self.tSavedData[KEY_ITEMS][reverseContainerCache[timeKey]]["cont"][itemid]=self.tSavedData[KEY_ITEMS][reverseContainerCache[timeKey]]["cont"][itemid]+1
				end
			end
			if reverseSalvageCache[timeKey] ~= nil then
				if self.tSavedData[KEY_ITEMS][reverseSalvageCache[timeKey]]["salv2"] == nil then
					self.tSavedData[KEY_ITEMS][reverseSalvageCache[timeKey]]["salv2"]={}
				end
				if self.tSavedData[KEY_ITEMS][reverseSalvageCache[timeKey]]["salv2"][itemid] == nil then
					self.tSavedData[KEY_ITEMS][reverseSalvageCache[timeKey]]["salv2"][itemid]=1
				else
					self.tSavedData[KEY_ITEMS][reverseSalvageCache[timeKey]]["salv2"][itemid]=self.tSavedData[KEY_ITEMS][reverseSalvageCache[timeKey]]["salv2"][itemid]+1
				end
			end
		end
	end
end

function Jabbithole:OnItemRemoved(item)
	self:SetupInternals()
	if item then
		self:AddItem(item, -1, nil,true)
		local timeKey=math.floor(GameLib.GetGameTime()*10)
		itemid=self:MakeItemId(item)
		if item:CanAutoSalvage() then
			reverseContainerCache[timeKey]=itemid
			if containerCache[timeKey] ~= nil then
				if self.tSavedData[KEY_ITEMS][itemid]["cont"] == nil then
					self.tSavedData[KEY_ITEMS][itemid]["cont"]={}
				end
				for idx,itmid in pairs(containerCache[timeKey] or {}) do
					if self.tSavedData[KEY_ITEMS][itemid]["cont"][itmid] == nil then
						self.tSavedData[KEY_ITEMS][itemid]["cont"][itmid]=1
					else
						self.tSavedData[KEY_ITEMS][itemid]["cont"][itmid]=self.tSavedData[KEY_ITEMS][itemid]["cont"][itmid]+1
					end
				end
			end
		else
			if item:CanSalvage() then
				reverseSalvageCache[timeKey]=itemid
				if salvageCache[timeKey] ~= nil then
					if self.tSavedData[KEY_ITEMS][itemid]["salv2"] == nil then
						self.tSavedData[KEY_ITEMS][itemid]["salv2"]={}
					end
					for idx,itmid in pairs(salvageCache[timeKey] or {}) do
						if self.tSavedData[KEY_ITEMS][itemid]["salv2"][itmid] == nil then
							self.tSavedData[KEY_ITEMS][itemid]["salv2"][itmid]=1
						else
							self.tSavedData[KEY_ITEMS][itemid]["salv2"][itmid]=self.tSavedData[KEY_ITEMS][itemid]["salv2"][itmid]+1
						end
					end
				end				
			else
				--recipes
				local di=item:GetDetailedInfo().tPrimary
				if di and di.tCharge then
					if di.eFamily==19 then --schematic
						teachesCache[timeKey]=itemid
						if reverseTeachesCache[timeKey] ~= nil then
							if self.tSavedData[KEY_ITEMS][teachesCache[timeKey]]["teaches"] == nil then
								self.tSavedData[KEY_ITEMS][teachesCache[timeKey]]["teaches"]={}
							end
							self.tSavedData[KEY_ITEMS][teachesCache[timeKey]]["teaches"][reverseTeachesCache[timeKey]["id"]]=reverseTeachesCache[timeKey]["p"]
						end
					end
					if di.eType==332 then  --dye
						dyeCache[timeKey]=itemid
						if reverseDyeCache[timeKey] ~= nil then
							self.tSavedData[KEY_ITEMS][dyeCache[timeKey]]["teachesDye"]=reverseDyeCache[timeKey]
						end
					end
					
				end
			end
		end
	end
end

function Jabbithole:OnCraftingSchematicLearned(profId, schId)
	self:SetupInternals()
	local timeKey=math.floor(GameLib.GetGameTime()*10)
	if teachesCache[timeKey] ~= nil then
		if self.tSavedData[KEY_ITEMS][teachesCache[timeKey]]["teaches"] == nil then
			self.tSavedData[KEY_ITEMS][teachesCache[timeKey]]["teaches"]={}
		end
		self.tSavedData[KEY_ITEMS][teachesCache[timeKey]]["teaches"][schId]=profId
	end
	reverseTeachesCache[timeKey]={}
	reverseTeachesCache[timeKey]["id"]=schId
	reverseTeachesCache[timeKey]["p"]=profId
	self:AddSchematic(schId, profId)
end

function Jabbithole:OnDyeLearned()
	local timeKey=math.floor(GameLib.GetGameTime()*10)
	
	local allDyes=GameLib.GetKnownDyes()
	local newDye=nil

	for idx,dye in pairs(allDyes or {}) do
		local searchFor=dye
		for oidx,odye in pairs(self.knownDyes or {}) do
			if searchFor and searchFor.nId==odye.nId then
				searchFor=nil
			end
		end
		if searchFor then
			newDye=searchFor
		end
	end
	
	self.knownDyes=allDyes
	
	if newDye then
		if dyeCache[timeKey] ~= nil then
			self.tSavedData[KEY_ITEMS][dyeCache[timeKey]]["teachesDye"]=newDye.nId
		end
		reverseDyeCache[timeKey]=newDye.nId
	end
end

function Jabbithole:OnUnitCreated(unit)
	self:SetupInternals()
	if unit and unit:IsValid() and not unit:IsACharacter() then
		if unit:GetType() == "PinataLoot" then
			droppedItems[unit:GetName()]=true
			local loot=unit:GetLoot()
			if loot then
				if loot.itemLoot then
					if loot.idOwner then
						if unitCache[loot.idOwner] then
							self:AddCreature(unitCache[loot.idOwner],true)
							self:AddItem(loot.itemLoot,-1,unitCache[loot.idOwner],true)
						end
					end
				end
			end
		else
			unitCache[unit:GetId()]=unit
		end
	end
end 

function Jabbithole:OnUnitDestroyed(unit)
	self:SetupInternals()
	if unit then
		if unit:GetType() == "PinataLoot" then
			local loot=unit:GetLoot()
			if loot then
				if loot.itemLoot then
-- save items looted by others too
--					if justLootedCache[loot.item:GetItemId()] then
					if loot.idOwner then
						if unitCache[loot.idOwner] then
							self:AddCreature(unitCache[loot.idOwner],true)
							self:AddItem(loot.itemLoot ,-1,unitCache[loot.idOwner],true)
						else
							self:AddItem(loot.itemLoot ,-1,nil,true)
						end
					else
						self:AddItem(loot.itemLoot ,-1,nil,true)
					end
				end
			end
		end
	end
end 

function Jabbithole:OnDialog_ShowState(eState, queCurrent)
	self:SetupInternals()
	if queCurrent then
		if eState == DialogSys.DialogState_QuestAccept or eState == DialogSys.DialogState_TopicChoice then
			local unitNpc = DialogSys.GetNPC()
			local unitComm = DialogSys.GetCommCreatureId()
			
			self:AddQuest(queCurrent, unitNpc, unitComm, true)		
		end
		if eState == DialogSys.DialogState_QuestComplete then
			if self.tSavedData[KEY_QUESTS] ~=nil then
				local unitNpc = DialogSys.GetNPC()
				local unitComm = DialogSys.GetCommCreatureId()
	
				local id=queCurrent:GetId()
				if self.tSavedData[KEY_QUESTS][id] ~= nil then
					if unitNpc and unitNpc:IsValid() then
						local npc=self:MakeCreatureId(unitNpc)

						if self.tSavedData[KEY_QUESTS][id]["finish"]==nil then
							self.tSavedData[KEY_QUESTS][id]["finish"]={}
						end
						if self.tSavedData[KEY_QUESTS][id]["finish"][npc]==nil then
							self.tSavedData[KEY_QUESTS][id]["finish"][npc]={}
						end
						
						self.tSavedData[KEY_QUESTS][id]["finish"][npc][unitNpc:GetPosition().x.."/"..unitNpc:GetPosition().y.."/"..unitNpc:GetPosition().z]=1
						self:AddCreature(unitNpc)
					end
					if unitComm then
						self.tSavedData[KEY_QUESTS][id]["callfinish"]=true
					end
				end
			end
		end
	end
end

function Jabbithole:SetupInternals()
	self.tPlayer = GameLib.GetPlayerUnit()
	if self.tPlayer ~= nil then
		self.tPlayerFaction = self.tPlayer:GetFaction()
	end
	self.tZone=GameLib.GetCurrentZoneMap()
	self.worldId=GameLib.GetCurrentWorldId()
	self.worldZoneId=GameLib.GetCurrentZoneId()
	if self.tZone==nil then
		self.tZone={
			id=0,
			strName="unspecified",
			continentId=0,
			strFolder="",
			fNorth=0,
			fEast=0,
			fSouth=0,
			fWest=0
		}
	end
end

function Jabbithole:OnSystemBeginDragDrop(wndSource, strType, iData)
	self:SetupInternals()
	if strType == "DDBagItem" then
		local item=Item.GetItemFromInventoryLoc(iData)
		if item then
			self:AddItem(item,-1,nil,true)
		end
	end
end

function Jabbithole:OnSubZoneChanged(id,name)
	self:SetupInternals()
--	self:AddPathEpisode(PlayerPathLib.GetCurrentEpisode())

	self:AddZone(self.tZone)

	self:AddPathEpisode(PlayerPathLib.GetPathEpisodeForZone())
	
--	GroupLib.InGroup()
--	GroupLib.InInstance()
	
--[[	local is = GameLib.GetInstanceSettings()
	if is then
		if is.eWorldDifficulty == GroupLib.Difficulty.Veteran or GroupLib.GetInstanceDifficulty() == GroupLib.Difficulty.Veteran then
			bVeteranMode = true
		else
			bVeteranMode = false
		end
	else
		bVeteranMode = false
	end]]
	bVeteranMode = (GameLib.GetWorldDifficulty()==GroupLib.Difficulty.Veteran)
	
	unitCache = {}
	
	self:SaveHousingStuff()
end

function Jabbithole:OnWorldChanged()
	self:OnSubZoneChanged()
end

function Jabbithole:OnCanVacuumChanged(bCanVacuum)
	self:SetupInternals()
	if not bCanVacuum then
		droppedItems = {}
		justLootedCache = {}	
	end
end


function Jabbithole:OnPlayerTitleUpdate()
	self:SetupInternals()
	local tTitles = CharacterTitle.GetAvailableTitles()
	for idx, titleCurr in pairs(tTitles) do
		self:AddTitle(titleCurr:GetTitle(), titleCurr:GetCategory(), titleCurr:GetForUnit())
	end
end

function Jabbithole:OnVendorItemsUpdated()
	self:SetupInternals()
	if self.vendorUnit then
		local items = self.vendorUnit:GetVendorItems()
		if items and #items>0 then
			for idx, item in pairs(items) do
				if item.itemData then
					self:AddItem(item.itemData,-1,nil,true)
					self:AddVendorItem(self.vendorUnit,item)
				end
				if item.splData then
					self:AddSpell(item.splData)
					self:AddVendorSpell(self.vendorUnit,item)
				end
			end
		end
	end
end
function Jabbithole:OnInvokeVendorWindow(unit)
	self:SetupInternals()
	if unit and unit:IsValid() then
		self:AddCreature(unit)
		self.vendorUnit=unit
	end
end

function Jabbithole:OnTargetUnitChanged(unit)
	self:SetupInternals()
	self:AddZone(self.tZone)
	if unit then
		self:AddCreature(unit)
	end 
end

function Jabbithole:OnChallengeActivate(chg)
	self:SetupInternals()
	self:AddChallenge(chg)
end

--[[ obsolete in reloaded
function Jabbithole:OnChallengeRewardListReady(chgid,tier)
	self:SetupInternals()
	for idx,reward in pairs(ChallengesLib.GetRewardList(chgid) or {}) do
		self:AddChallengeReward(chgid,reward)
	end
end
]]

function Jabbithole:OnPlayerPathMissionUnlocked(pm)
	self:SetupInternals()
	if pm then
		self:AddPathMission(PlayerPathLib.GetPlayerPathType(),pm,true)
	end
end

function Jabbithole:OnPlayerPathMissionDeactivate(pm)
	self:SetupInternals()
	if pm then
		self:AddPathMission(PlayerPathLib.GetPlayerPathType(),pm,true)
	end
end
	
function Jabbithole:OnPublicEventStart(pe)
	self:SetupInternals()
	if pe ~= nil and #pe:GetObjectives() > 0 then
		if pe:IsActive() then
			self:AddPublicEvent(pe)
		end
	end
end

function Jabbithole:OnPublicEventInitiateVote()
	self:SetupInternals()
	local tVoteData=PublicEvent.GetActiveVote()
	if tVoteData then
		local peid=-1
		for idx,e in pairs(PublicEvent.GetActiveEvents() or {}) do
			if e:GetEventType() ~= 11 then
				peid=self:FindPublicEvent(e)
			end
		end
		if peid ~= -1 then
			if self.tSavedData[KEY_PUBLIC_EVENT][peid]["realzone"]==nil then
				self.tSavedData[KEY_PUBLIC_EVENT][peid]["realzone"]=self.tZone.id
			else
				if self.tSavedData[KEY_PUBLIC_EVENT][peid]["realzone"] ~= self.tZone.id then
					self.tSavedData[KEY_PUBLIC_EVENT][peid]["realzone"] = -1
				end
			end
			for idx,opt in pairs(tVoteData.arOptions or {}) do
				self:AddPublicEventMission(peid, opt.strLabel, opt.strChoiceDescription)
			end
		end
	end
end

function Jabbithole:OnCombatLogDamage(tEventArgs)
	if tEventArgs.unitCaster then
		self:AddCreature(tEventArgs.unitCaster)
	end
	if tEventArgs.splCallingSpell then
		self:AddSpell(tEventArgs.splCallingSpell)
	end
	self:AddCreatureSpellRelation(tEventArgs.unitCaster, tEventArgs.splCallingSpell)
end

function Jabbithole:OnDatacubeUpdated(id, bIsVolume)
	self:SetupInternals()
	if not id then
		return
	end

	local tDatacube = DatacubeLib.GetLastUpdatedDatacube(id, bIsVolume)
	self:AddDatacube(tDatacube.nDatacubeId, bIsVolume, tDatacube)
end

function Jabbithole:OnHousingPlugItemsUpdated()
	for idx, plot in pairs(HousingLib.GetVendorList() or {}) do
		self:AddHousingPlot(plot)
	end
end

function Jabbithole:OnItemSentToCrate(itemSentToCrate, nCount)
	if itemSentToCrate and nCount>0 then
		self:AddItem(itemSentToCrate,-1,nil,true)
		self:SaveHousingStuff()
	end
end

function Jabbithole:OnContractBoardOpen()
	self:AddContracts()
	self:AddContractRewards()
end

local function CountKeys(tbl)
	local c=0
	for k,v in pairs(tbl or {}) do
		c=c+1
	end
	return c
end

function Jabbithole:OnJabbitholeOn()
	self.wndMain:Show(not self.wndMain:IsVisible())
	if self.wndMain:IsVisible() then
		self.wndMain:FindChild("NumItems"):SetText(CountKeys(self.tSavedData[KEY_ITEMS]))
		self.wndMain:FindChild("NumCreatures"):SetText(CountKeys(self.tSavedData[KEY_CREATURES]))
		self.wndMain:FindChild("NumAchis"):SetText(CountKeys(self.tSavedData[KEY_ACHI]))
		self.wndMain:FindChild("NumSch"):SetText(CountKeys(self.tSavedData[KEY_SCHEMATIC]))
		self.wndMain:FindChild("NumQuests"):SetText(CountKeys(self.tSavedData[KEY_QUESTS]))
		self.wndMain:FindChild("NumPMs"):SetText(CountKeys(self.tSavedData[KEY_PATH_MISSION]))
		self.wndMain:FindChild("NumSpells"):SetText(CountKeys(self.tSavedData[KEY_SPELLS]))
		self.wndMain:FindChild("NumChg"):SetText(CountKeys(self.tSavedData[KEY_CHALLENGE]))
		self.wndMain:FindChild("NumPEs"):SetText(CountKeys(self.tSavedData[KEY_PUBLIC_EVENT]))
	end
end

function Jabbithole:OnPlayerExit()
	self:AddCharacter()
end

function Jabbithole:OnRestore(eLevel, tData)
    if eLevel == GameLib.CodeEnumAddonSaveLevel.General then
	    self.tSavedData = tData
		if self.tSavedData["deprecates"]~=deprecates then
			self.tSavedData={}
		end
		if self.tSavedData["l"]~=Apollo.GetString("CRB_Yes") then
			self.tSavedData={}
		end
		
		self.NumItems=CountKeys(self.tSavedData[KEY_ITEMS])
		self.NumSchematics=CountKeys(self.tSavedData[KEY_SCHEMATIC])
		self.NumSpells=CountKeys(self.tSavedData[KEY_SPELLS])
		self.NumCreatures=CountKeys(self.tSavedData[KEY_CREATURES])
		self.NumQuests=CountKeys(self.tSavedData[KEY_QUESTS])
		
		if tData.lastULWarn then
			lastUploadWarning = tData.lastULWarn
		end
    end
	if eLevel == GameLib.CodeEnumAddonSaveLevel.Character then
		if tData and tData.lastTSTS then
			lastTradeskillSaveTime = tData.lastTSTS
		end
	end
	if eLevel == GameLib.CodeEnumAddonSaveLevel.Account then
		if tData then
			self.accountVars = tData
		end
	end
end

function Jabbithole:OnSave(eLevel)
    if eLevel == GameLib.CodeEnumAddonSaveLevel.General then
		self.tSavedData["v"]=self.tSavedData["v"]+1
		self.tSavedData["l"]=Apollo.GetString("CRB_Yes")
		self.tSavedData["vv"]=VERSION
		if self.tSavedData["firstdata"]==nil then
			self.tSavedData["firstdata"]=os.time()
		end
		local vi=GameLib.GetVersionInfo()
		if self.tSavedData["firstversion"]==nil then
			self.tSavedData["firstversion"]=vi.nVersion.."."..vi.nVersionMajor.."."..vi.nVersionMinor.."."..vi.nBuildNumber
		end
		self.tSavedData["lastdata"]=os.time()
		self.tSavedData["lastversion"]=vi.nVersion.."."..vi.nVersionMajor.."."..vi.nVersionMinor.."."..vi.nBuildNumber
		self.tSavedData["deprecates"]=deprecates
		self.tSavedData["lastULWarn"]=lastUploadWarning
		
		--self:AddCharacter()
		
	    return self.tSavedData 
    end
    if eLevel == GameLib.CodeEnumAddonSaveLevel.Character then
		return {lastTSTS = lastTradeskillSaveTime}
	end
    if eLevel == GameLib.CodeEnumAddonSaveLevel.Account then
		return self.accountVars 
	end
	return nil
end
-----------------------------------------------------------------------------------------------
-- JabbitholeForm Functions
-----------------------------------------------------------------------------------------------
-- when the OK button is clicked
function Jabbithole:OnOK()
	self.wndMain:Show(false) -- hide the window
end

-- when the Cancel button is clicked
function Jabbithole:OnCancel()
	self.wndMain:Show(false) -- hide the window
end

function Jabbithole:OnWarningOkClicked()
	lastUploadWarning = os.time()
	self.wndWarning:Show(false) -- hide the window
end

function Jabbithole:OnExportShow()
	self.wndExport:Show(true)
	self.wndExport:ToFront()
end

function Jabbithole:OnExportCancel()
	self.wndExport:Show(false) -- hide the window
	self.wndExport:FindChild("EditBoxExportKey"):SetText("")
	self.wndExport:FindChild("EditBoxCharacterData"):SetText("")
end

function Jabbithole:OnExportExport()
	local key=self.wndExport:FindChild("EditBoxExportKey"):GetText()
	if key and key~="" then

		local player = GameLib.GetPlayerUnit()
		if player and player:GetLevel()>9 then
			local id = player:GetName() ..'@'.. GameLib.GetRealmName()
			local data={}
			data.char=self.tSavedData[KEY_CHARACTERS][id]
			if data.char then
				--TODO
				--add items
				data.key=key
				data.items={}
				for _, slot in pairs(data.char.runes) do
					for _, rune in pairs(slot) do
						if rune and rune ~= "" then
							local splits=explode_string("/",rune)
							data.items[splits[1]]=self.tSavedData[KEY_ITEMS][splits[1]]
						end
					end
				end
				for _, itemid in pairs(data.char.items) do
					data.items[itemid]=self.tSavedData[KEY_ITEMS][itemid]
				end
			
				local packed=JSON:encode(data)
--				local packed=CompressHuffman(JSON:encode(data))
				local sliced=split_string(packed,7997)
				local encoded=''				
				for idx, line in ipairs(sliced) do
--				   encoded=encoded..encode64(line).."\n"
				   encoded=encoded..encode64(jh_crypt(key,line)).."\n"
				end
				
				self.wndExport:FindChild("EditBoxCharacterData"):SetText(encoded)
				
				self.wndExport:FindChild("CopyToClipboardButton"):SetActionData(GameLib.CodeEnumConfirmButtonType.CopyToClipboard, encoded)
			end
		end
	else
		self.wndExport:FindChild("EditBoxExportKey"):SetFocus()
		Sound.Play(107)
	end
end

function Jabbithole:OnExportCopy()
end

function Jabbithole:OnWarnButtonToggle( wndHandler, wndControl, eMouseButton )
--	SendVarToRover('x',wndControl)
	self.accountVars.warnUpload = self.wndMain:FindChild("WarnButton"):IsChecked()
end

function Jabbithole:OnArmoryButtonToggle( wndHandler, wndControl, eMouseButton )
	self.accountVars.saveArmory = self.wndMain:FindChild("ArmoryButton"):IsChecked()
	if self.accountVars.saveArmory == false then
		self.tSavedData[KEY_CHARACTERS]=nil
	end
end

function nlFix(s)
	local ret=''
	local trash=0
	if s then
		ret,trash = s:gsub('\n','\\n')
	end
	return ret
end

function split_string(str, max_line_length)
   local lines = {}
   local line
   str:gsub('(%s*)(%S+)', 
      function(spc, word) 
         if not line or #line + #spc + #word > max_line_length then
            table.insert(lines, line)
            line = word
         else
            line = line..spc..word
         end
      end
   )
   table.insert(lines, line)
   return lines
end

-- explode(seperator, string)
function explode_string(d,p)
  local t, ll
  t={}
  ll=0
  if(#p == 1) then return {p} end
    while true do
      l=string.find(p,d,ll,true) -- find the next d in the string
      if l~=nil then -- if "not not" found then..
        table.insert(t, string.sub(p,ll,l-1)) -- Save it in our array.
        ll=l+1 -- save just after where we found it for searching next time.
      else
        table.insert(t, string.sub(p,ll)) -- Save what's left in our array.
        break -- Break at end, as it should be, according to the lua manual.
      end
    end
  return t
end

-----------------------------------------------------------------------------------------------
-- Jabbithole Instance
-----------------------------------------------------------------------------------------------
local JabbitholeInst = Jabbithole:new()
JabbitholeInst:Init()
